"use strict";
(() => {
  // extension/types.ts
  var MAX_USERS = 5;
  var DEFAULT_AVATARS = [
    "\u{1F600}",
    "\u{1F60E}",
    "\u{1F913}",
    "\u{1F973}",
    "\u{1F929}",
    "\u{1F63A}",
    "\u{1F436}",
    "\u{1F98A}",
    "\u{1F43C}",
    "\u{1F981}",
    "\u{1F680}",
    "\u2B50",
    "\u{1F308}",
    "\u{1F3A8}",
    "\u{1F3AE}",
    "\u{1F996}",
    "\u{1F995}",
    "\u{1F409}",
    "\u{1F525}",
    "\u26A1"
  ];

  // extension/popup/popup.ts
  var API_BASE_URLS = [
    "http://localhost:5180",
    "http://192.168.7.139:5180",
    "https://typetube.vercel.app"
  ];
  var enabledToggle = document.getElementById("enabled-toggle");
  var statWpm = document.getElementById("stat-wpm");
  var statAccuracy = document.getElementById("stat-accuracy");
  var statChallenges = document.getElementById("stat-challenges");
  var openDashboard = document.getElementById("open-dashboard");
  var currentUserAvatar = document.getElementById("current-user-avatar");
  var currentUserName = document.getElementById("current-user-name");
  var userSwitchBtn = document.getElementById("user-switch-btn");
  var userModal = document.getElementById("user-modal");
  var modalClose = document.getElementById("modal-close");
  var userList = document.getElementById("user-list");
  var newUserName = document.getElementById("new-user-name");
  var avatarPicker = document.getElementById("avatar-picker");
  var createUserBtn = document.getElementById("create-user-btn");
  var createUserSection = document.getElementById("create-user-section");
  var goalRingProgress = document.getElementById("goal-ring-progress");
  var goalPercent = document.getElementById("goal-percent");
  var goalCharsTyped = document.getElementById("goal-chars-typed");
  var goalCharsTotal = document.getElementById("goal-chars-total");
  var goalStatus = document.getElementById("goal-status");
  var healthHeader = document.getElementById("health-header");
  var healthToggleIcon = document.getElementById("health-toggle-icon");
  var healthContent = document.getElementById("health-content");
  var healthDashboard = document.getElementById("health-dashboard");
  var healthUser = document.getElementById("health-user");
  var healthSync = document.getElementById("health-sync");
  var healthLastSync = document.getElementById("health-last-sync");
  var healthVersion = document.getElementById("health-version");
  var testConnectionBtn = document.getElementById("test-connection-btn");
  var refreshHealthBtn = document.getElementById("refresh-health-btn");
  var syncNowBtn = document.getElementById("sync-now-btn");
  var migrateDataBtn = document.getElementById("migrate-data-btn");
  var healthMessage = document.getElementById("health-message");
  var selectedAvatar = "";
  var currentUser = null;
  var healthExpanded = false;
  async function loadCurrentUser() {
    const result = await chrome.runtime.sendMessage({ type: "GET_CURRENT_USER" });
    currentUser = result?.user || null;
    if (currentUser) {
      currentUserAvatar.textContent = currentUser.avatar;
      currentUserName.textContent = currentUser.name;
    } else {
      currentUserAvatar.textContent = "\u{1F464}";
      currentUserName.textContent = "Select a user";
    }
  }
  async function openUserModal() {
    const result = await chrome.runtime.sendMessage({ type: "GET_USER_LIST" });
    const users = result?.users || [];
    renderUserList(users);
    setupAvatarPicker();
    if (users.length >= MAX_USERS) {
      createUserSection.style.display = "none";
    } else {
      createUserSection.style.display = "block";
    }
    userModal.style.display = "flex";
  }
  function closeUserModal() {
    userModal.style.display = "none";
    newUserName.value = "";
    selectedAvatar = "";
    updateCreateUserButton();
  }
  function renderUserList(users) {
    if (users.length === 0) {
      userList.innerHTML = '<p class="no-users">No users yet. Create one below!</p>';
      return;
    }
    userList.innerHTML = users.map(
      (user) => `
    <div class="user-item ${currentUser?.id === user.id ? "active" : ""}" data-user-id="${user.id}">
      <span class="user-item-avatar">${user.avatar}</span>
      <span class="user-item-name">${user.name}</span>
      ${currentUser?.id === user.id ? '<span class="user-item-badge">Current</span>' : ""}
    </div>
  `
    ).join("");
    const userItems = userList.querySelectorAll(".user-item");
    userItems.forEach((item) => {
      item.addEventListener("click", () => {
        const userId = item.dataset.userId;
        if (userId) {
          handleUserSwitch(userId);
        }
      });
    });
  }
  function setupAvatarPicker() {
    avatarPicker.innerHTML = DEFAULT_AVATARS.map(
      (emoji) => `
    <button class="avatar-option" data-avatar="${emoji}">${emoji}</button>
  `
    ).join("");
    const avatarOptions = avatarPicker.querySelectorAll(".avatar-option");
    avatarOptions.forEach((option) => {
      option.addEventListener("click", (e) => {
        e.preventDefault();
        const avatar = option.dataset.avatar;
        if (avatar) {
          handleAvatarSelect(avatar);
        }
      });
    });
  }
  function handleAvatarSelect(avatar) {
    selectedAvatar = avatar;
    const avatarOptions = avatarPicker.querySelectorAll(".avatar-option");
    avatarOptions.forEach((option) => {
      if (option.dataset.avatar === avatar) {
        option.classList.add("selected");
      } else {
        option.classList.remove("selected");
      }
    });
    updateCreateUserButton();
  }
  function updateCreateUserButton() {
    const name = newUserName.value.trim();
    createUserBtn.disabled = !name || !selectedAvatar;
  }
  async function handleUserSwitch(userId) {
    await chrome.runtime.sendMessage({ type: "SWITCH_USER", userId });
    await loadCurrentUser();
    await loadStats();
    closeUserModal();
  }
  async function handleCreateUser() {
    const name = newUserName.value.trim();
    if (!name || !selectedAvatar) return;
    const result = await chrome.runtime.sendMessage({
      type: "CREATE_USER",
      name,
      avatar: selectedAvatar
    });
    if (result?.success) {
      await handleUserSwitch(result.user.id);
    }
  }
  async function loadSettings() {
    const settings = await chrome.runtime.sendMessage({ type: "GET_SETTINGS" });
    if (settings) {
      enabledToggle.checked = settings.enabled !== false;
    }
  }
  async function saveSettings() {
    const settings = {
      enabled: enabledToggle.checked
    };
    await chrome.runtime.sendMessage({ type: "SAVE_SETTINGS", settings });
  }
  async function loadStats() {
    if (!currentUser) {
      statWpm.textContent = "0";
      statAccuracy.textContent = "0%";
      statChallenges.textContent = "0";
      return;
    }
    const result = await chrome.runtime.sendMessage({
      type: "GET_STATS",
      userId: currentUser.id
    });
    const stats = result?.typingStats || [];
    const today = /* @__PURE__ */ new Date();
    today.setHours(0, 0, 0, 0);
    const todayTimestamp = today.getTime();
    const todayStats = stats.filter((s) => s.timestamp >= todayTimestamp);
    if (todayStats.length > 0) {
      const avgWpm = Math.round(
        todayStats.reduce((sum, s) => sum + s.wpm, 0) / todayStats.length
      );
      const avgAccuracy = Math.round(
        todayStats.reduce((sum, s) => sum + s.accuracy, 0) / todayStats.length
      );
      statWpm.textContent = String(avgWpm);
      statAccuracy.textContent = `${avgAccuracy}%`;
      statChallenges.textContent = String(todayStats.length);
    } else {
      statWpm.textContent = "0";
      statAccuracy.textContent = "0%";
      statChallenges.textContent = "0";
    }
  }
  function handleOpenDashboard(e) {
    e.preventDefault();
    chrome.tabs.create({ url: "http://localhost:5173" });
  }
  enabledToggle.addEventListener("change", saveSettings);
  openDashboard.addEventListener("click", handleOpenDashboard);
  userSwitchBtn.addEventListener("click", openUserModal);
  modalClose.addEventListener("click", closeUserModal);
  newUserName.addEventListener("input", updateCreateUserButton);
  createUserBtn.addEventListener("click", handleCreateUser);
  userModal.addEventListener("click", (e) => {
    if (e.target === userModal) {
      closeUserModal();
    }
  });
  function updateHealthItem(element, status, value) {
    const dot = element.querySelector(".health-dot");
    const valueEl = element.querySelector(".health-value");
    dot.classList.remove("green", "yellow", "red", "gray");
    dot.classList.add(status);
    valueEl.textContent = value;
  }
  function formatRelativeTime(timestamp) {
    const now = Date.now();
    const diff = now - timestamp;
    if (diff < 6e4) {
      return "Just now";
    } else if (diff < 36e5) {
      const mins = Math.floor(diff / 6e4);
      return `${mins} min${mins > 1 ? "s" : ""} ago`;
    } else if (diff < 864e5) {
      const hours = Math.floor(diff / 36e5);
      return `${hours} hour${hours > 1 ? "s" : ""} ago`;
    } else {
      const days = Math.floor(diff / 864e5);
      return `${days} day${days > 1 ? "s" : ""} ago`;
    }
  }
  async function checkApiConnectivity() {
    for (const url of API_BASE_URLS) {
      try {
        const response = await fetch(`${url}/api/users`, {
          method: "GET",
          headers: { "Accept": "application/json" }
        });
        if (response.ok) {
          return { connected: true, url };
        }
      } catch (e) {
      }
    }
    return { connected: false, url: null, error: "No server reachable" };
  }
  async function getSyncStatus() {
    try {
      const result = await chrome.runtime.sendMessage({ type: "GET_SYNC_STATUS" });
      return result || { pendingCount: 0, lastSyncAttempt: null };
    } catch {
      return { pendingCount: 0, lastSyncAttempt: null };
    }
  }
  async function getPendingSessions() {
    const status = await getSyncStatus();
    return status.pendingCount;
  }
  async function getLastSyncTimestamp() {
    const status = await getSyncStatus();
    return status.lastSyncAttempt;
  }
  function getExtensionVersion() {
    return chrome.runtime.getManifest().version;
  }
  async function runHealthCheck() {
    updateHealthItem(healthDashboard, "gray", "Checking...");
    updateHealthItem(healthUser, "gray", "Checking...");
    updateHealthItem(healthSync, "gray", "Checking...");
    updateHealthItem(healthLastSync, "gray", "Checking...");
    const [dashboardStatus, pendingSessions, lastSync] = await Promise.all([
      checkApiConnectivity(),
      getPendingSessions(),
      getLastSyncTimestamp()
    ]);
    const version = getExtensionVersion();
    if (dashboardStatus.connected) {
      const shortUrl = dashboardStatus.url?.replace("http://", "") || "";
      updateHealthItem(healthDashboard, "green", `Connected (${shortUrl})`);
    } else {
      updateHealthItem(healthDashboard, "red", "Offline");
    }
    if (currentUser) {
      updateHealthItem(healthUser, "green", currentUser.name);
    } else {
      updateHealthItem(healthUser, "yellow", "Not set");
    }
    if (pendingSessions === 0) {
      updateHealthItem(healthSync, "green", "None");
    } else {
      updateHealthItem(healthSync, "yellow", `${pendingSessions} session${pendingSessions > 1 ? "s" : ""}`);
    }
    if (lastSync) {
      updateHealthItem(healthLastSync, "green", formatRelativeTime(lastSync));
    } else {
      updateHealthItem(healthLastSync, "gray", "Never");
    }
    updateHealthItem(healthVersion, "green", `v${version}`);
    return {
      dashboard: dashboardStatus,
      user: { loaded: !!currentUser, name: currentUser?.name || null },
      pendingSessions,
      lastSync,
      version
    };
  }
  function showHealthMessage(message, type) {
    healthMessage.textContent = message;
    healthMessage.className = `health-message ${type}`;
    healthMessage.style.display = "block";
  }
  function hideHealthMessage() {
    healthMessage.style.display = "none";
  }
  function toggleHealthSection() {
    healthExpanded = !healthExpanded;
    if (healthExpanded) {
      healthContent.style.display = "block";
      healthToggleIcon.classList.add("expanded");
      runHealthCheck();
    } else {
      healthContent.style.display = "none";
      healthToggleIcon.classList.remove("expanded");
      hideHealthMessage();
    }
  }
  async function testConnection() {
    testConnectionBtn.disabled = true;
    testConnectionBtn.textContent = "Testing...";
    hideHealthMessage();
    const results = [];
    for (const url of API_BASE_URLS) {
      try {
        const startTime = Date.now();
        const response = await fetch(`${url}/api/users`, {
          method: "GET",
          headers: { "Accept": "application/json" }
        });
        const elapsed = Date.now() - startTime;
        if (response.ok) {
          results.push(`${url}: OK (${elapsed}ms)`);
        } else {
          results.push(`${url}: HTTP ${response.status}`);
        }
      } catch (e) {
        const error = e instanceof Error ? e.message : "Unknown error";
        results.push(`${url}: ${error}`);
      }
    }
    const anySuccess = results.some((r) => r.includes("OK"));
    if (anySuccess) {
      showHealthMessage(results.join("\n"), "success");
    } else {
      showHealthMessage(`All servers unreachable:
${results.join("\n")}`, "error");
    }
    testConnectionBtn.disabled = false;
    testConnectionBtn.textContent = "Test Connection";
  }
  async function refreshHealth() {
    refreshHealthBtn.disabled = true;
    refreshHealthBtn.textContent = "Refreshing...";
    hideHealthMessage();
    await runHealthCheck();
    refreshHealthBtn.disabled = false;
    refreshHealthBtn.textContent = "Refresh";
  }
  async function syncNow() {
    syncNowBtn.disabled = true;
    syncNowBtn.textContent = "Syncing...";
    hideHealthMessage();
    try {
      const result = await chrome.runtime.sendMessage({ type: "SYNC_PENDING" });
      if (result.success) {
        if (result.synced > 0) {
          showHealthMessage(
            `Synced ${result.synced} session${result.synced > 1 ? "s" : ""}. ${result.remaining} remaining.`,
            "success"
          );
        } else if (result.remaining === 0) {
          showHealthMessage("No pending sessions to sync.", "info");
        } else {
          showHealthMessage(`Sync failed. ${result.remaining} sessions still pending.`, "error");
        }
      } else {
        showHealthMessage(`Sync failed: ${result.error || "Unknown error"}`, "error");
      }
      await runHealthCheck();
    } catch (e) {
      const error = e instanceof Error ? e.message : "Unknown error";
      showHealthMessage(`Sync failed: ${error}`, "error");
    }
    syncNowBtn.disabled = false;
    syncNowBtn.textContent = "Sync Now";
  }
  async function migrateData() {
    migrateDataBtn.disabled = true;
    migrateDataBtn.textContent = "Migrating...";
    hideHealthMessage();
    try {
      const result = await chrome.runtime.sendMessage({ type: "MIGRATE_DATA" });
      if (result.success) {
        if (result.totalMigrated > 0) {
          showHealthMessage(
            `Migration complete! Migrated ${result.totalMigrated} session${result.totalMigrated > 1 ? "s" : ""}.`,
            "success"
          );
        } else {
          showHealthMessage("No data to migrate or already migrated.", "info");
        }
      } else {
        const failedMsg = result.totalFailed > 0 ? ` ${result.totalFailed} session${result.totalFailed > 1 ? "s" : ""} failed.` : "";
        showHealthMessage(`Migration issue: ${result.error || "Unknown error"}${failedMsg}`, "error");
      }
      await runHealthCheck();
    } catch (e) {
      const error = e instanceof Error ? e.message : "Unknown error";
      showHealthMessage(`Migration failed: ${error}`, "error");
    }
    migrateDataBtn.disabled = false;
    migrateDataBtn.textContent = "Migrate Data";
  }
  healthHeader.addEventListener("click", toggleHealthSection);
  testConnectionBtn.addEventListener("click", testConnection);
  refreshHealthBtn.addEventListener("click", refreshHealth);
  syncNowBtn.addEventListener("click", syncNow);
  migrateDataBtn.addEventListener("click", migrateData);
  async function loadDailyGoal() {
    if (!currentUser) {
      updateGoalDisplay({ charactersTyped: 0, goalChars: 500, charProgress: 0, isGoalMet: false });
      return;
    }
    try {
      const result = await chrome.runtime.sendMessage({
        type: "GET_STATS",
        userId: currentUser.id
      });
      const stats = result?.typingStats || [];
      const today = /* @__PURE__ */ new Date();
      today.setHours(0, 0, 0, 0);
      const todayTimestamp = today.getTime();
      const todayStats = stats.filter((s) => s.timestamp >= todayTimestamp);
      const charactersTyped = todayStats.reduce(
        (sum, s) => sum + (s.chars || s.totalCharacters || 0),
        0
      );
      const userSettings = await chrome.runtime.sendMessage({
        type: "GET_USER_SETTINGS",
        userId: currentUser.id
      });
      const goalChars = userSettings?.dailyGoalChars || 500;
      const charProgress = Math.min(100, Math.round(charactersTyped / goalChars * 100));
      const isGoalMet = charProgress >= 100;
      updateGoalDisplay({ charactersTyped, goalChars, charProgress, isGoalMet });
    } catch (error) {
      console.error("[Popup] Error loading daily goal:", error);
      updateGoalDisplay({ charactersTyped: 0, goalChars: 500, charProgress: 0, isGoalMet: false });
    }
  }
  function updateGoalDisplay(progress) {
    const circumference = 201;
    const offset = circumference - progress.charProgress / 100 * circumference;
    goalRingProgress.style.strokeDashoffset = String(offset);
    goalPercent.textContent = `${progress.charProgress}%`;
    goalCharsTyped.textContent = String(progress.charactersTyped);
    goalCharsTotal.textContent = String(progress.goalChars);
    if (progress.isGoalMet) {
      goalStatus.textContent = "Goal met! Great job!";
      goalStatus.classList.add("complete");
      goalRingProgress.style.stroke = "#22c55e";
    } else {
      goalStatus.textContent = "Keep typing!";
      goalStatus.classList.remove("complete");
      goalRingProgress.style.stroke = "#667eea";
    }
  }
  var brainrotHeader = document.getElementById("brainrot-header");
  var brainrotToggleIcon = document.getElementById("brainrot-toggle-icon");
  var brainrotContent = document.getElementById("brainrot-content");
  var anthropicApiKeyInput = document.getElementById("anthropic-api-key");
  var saveApiKeyBtn = document.getElementById("save-api-key-btn");
  var testApiKeyBtn = document.getElementById("test-api-key-btn");
  var apiKeyStatus = document.getElementById("api-key-status");
  var brainrotExpanded = false;
  function toggleBrainrotSection() {
    brainrotExpanded = !brainrotExpanded;
    brainrotContent.style.display = brainrotExpanded ? "block" : "none";
    brainrotToggleIcon.textContent = brainrotExpanded ? "\u25B2" : "\u25BC";
    if (brainrotExpanded) {
      loadApiKey();
    }
  }
  async function loadApiKey() {
    const result = await chrome.runtime.sendMessage({ type: "GET_ANTHROPIC_API_KEY" });
    if (result?.apiKey) {
      anthropicApiKeyInput.value = "\u2022\u2022\u2022\u2022\u2022\u2022\u2022\u2022\u2022\u2022\u2022\u2022" + result.apiKey.slice(-4);
      if (result.source === "server") {
        apiKeyStatus.textContent = "Using API key from server (.env)";
        apiKeyStatus.style.color = "#22c55e";
        anthropicApiKeyInput.disabled = true;
        anthropicApiKeyInput.placeholder = "Provided by server";
        saveApiKeyBtn.disabled = true;
        saveApiKeyBtn.textContent = "From Server";
      } else {
        apiKeyStatus.textContent = "Using local API key";
        apiKeyStatus.style.color = "#22c55e";
        anthropicApiKeyInput.disabled = false;
        saveApiKeyBtn.disabled = false;
        saveApiKeyBtn.textContent = "Save Key";
      }
    } else {
      anthropicApiKeyInput.value = "";
      anthropicApiKeyInput.disabled = false;
      anthropicApiKeyInput.placeholder = "sk-ant-...";
      saveApiKeyBtn.disabled = false;
      saveApiKeyBtn.textContent = "Save Key";
      apiKeyStatus.textContent = "No API key (add to .env or enter below)";
      apiKeyStatus.style.color = "#f97316";
    }
  }
  async function saveApiKey() {
    const apiKey = anthropicApiKeyInput.value.trim();
    if (apiKey.startsWith("\u2022\u2022\u2022\u2022")) {
      apiKeyStatus.textContent = "Key already saved";
      apiKeyStatus.style.color = "#22c55e";
      return;
    }
    if (!apiKey || !apiKey.startsWith("sk-ant-")) {
      apiKeyStatus.textContent = "Invalid key format (should start with sk-ant-)";
      apiKeyStatus.style.color = "#ef4444";
      return;
    }
    await chrome.runtime.sendMessage({ type: "SET_ANTHROPIC_API_KEY", apiKey });
    apiKeyStatus.textContent = "API key saved!";
    apiKeyStatus.style.color = "#22c55e";
    anthropicApiKeyInput.value = "\u2022\u2022\u2022\u2022\u2022\u2022\u2022\u2022\u2022\u2022\u2022\u2022" + apiKey.slice(-4);
  }
  async function testApiKey() {
    apiKeyStatus.textContent = "Testing...";
    apiKeyStatus.style.color = "#9ca3af";
    try {
      const result = await chrome.runtime.sendMessage({
        type: "CLASSIFY_VIDEOS",
        videos: [{
          videoId: "test",
          title: "Test Video",
          channelName: "Test Channel"
        }]
      });
      if (result?.success) {
        apiKeyStatus.textContent = "API key works!";
        apiKeyStatus.style.color = "#22c55e";
      } else {
        apiKeyStatus.textContent = "API error: " + (result?.error || "Unknown");
        apiKeyStatus.style.color = "#ef4444";
      }
    } catch (e) {
      apiKeyStatus.textContent = "Test failed: " + e.message;
      apiKeyStatus.style.color = "#ef4444";
    }
  }
  if (brainrotHeader) {
    brainrotHeader.addEventListener("click", toggleBrainrotSection);
  }
  if (saveApiKeyBtn) {
    saveApiKeyBtn.addEventListener("click", saveApiKey);
  }
  if (testApiKeyBtn) {
    testApiKeyBtn.addEventListener("click", testApiKey);
  }
  async function init() {
    await loadCurrentUser();
    await loadSettings();
    await loadStats();
    await loadDailyGoal();
  }
  init();
})();
