"use strict";
(() => {
  // extension/dashboard-bridge.ts
  window.__TYPETUBE_EXTENSION__ = true;
  window.addEventListener("message", async (event) => {
    if (event.source !== window) return;
    const message = event.data;
    if (!message || message.source !== "typetube-dashboard") return;
    console.log("[TypeTube Bridge] Received message:", message.type);
    try {
      let response;
      switch (message.type) {
        case "GET_EXTENSION_STATUS":
          response = { available: true };
          break;
        case "GET_STATS":
          const statsKey = message.userId ? `typingStats_${message.userId}` : "typingStats";
          const statsResult = await chrome.storage.local.get([statsKey]);
          response = { stats: statsResult[statsKey] || [] };
          break;
        case "GET_ALL_STATS":
          const allData = await chrome.storage.local.get(null);
          const allStats = {};
          for (const [key, value] of Object.entries(allData)) {
            if (key.startsWith("typingStats")) {
              allStats[key] = value;
            }
          }
          response = { stats: allStats };
          break;
        case "GET_SETTINGS":
          const settingsResult = await chrome.storage.sync.get([
            "difficulty",
            "wordsPerChallenge",
            "rewardDuration",
            "estimatedWPM"
          ]);
          response = { settings: settingsResult };
          break;
        case "SAVE_SETTINGS":
          await chrome.storage.sync.set(message.settings);
          response = { success: true };
          break;
        case "GET_USERS":
          const usersResult = await chrome.storage.local.get(["typetube_users"]);
          response = { users: usersResult.typetube_users || { users: [], currentUserId: null } };
          break;
        case "GET_CURRENT_USER":
          const currentResult = await chrome.storage.local.get(["typetube_users"]);
          const userData = currentResult.typetube_users || { users: [], currentUserId: null };
          const currentUser = userData.users.find((u) => u.id === userData.currentUserId);
          response = { user: currentUser || null };
          break;
        case "GET_LETTER_PROGRESS":
          const letterResult = await chrome.storage.local.get(["letterProgress"]);
          response = { letterProgress: letterResult.letterProgress || null };
          break;
        case "GET_USER_SETTINGS":
          if (message.userId) {
            const userSettingsKey = `userSettings_${message.userId}`;
            const userSettingsResult = await chrome.storage.local.get([userSettingsKey]);
            response = { settings: userSettingsResult[userSettingsKey] || {} };
          } else {
            response = { settings: {} };
          }
          break;
        case "SAVE_USER_SETTINGS":
          if (message.userId && message.settings) {
            const saveKey = `userSettings_${message.userId}`;
            await chrome.storage.local.set({ [saveKey]: message.settings });
            response = { success: true };
            try {
              const tabs = await chrome.tabs.query({ url: "*://www.youtube.com/*" });
              for (const tab of tabs) {
                if (tab.id) {
                  chrome.tabs.sendMessage(tab.id, { type: "SETTINGS_CHANGED", userId: message.userId }).catch(() => {
                  });
                }
              }
            } catch (e) {
              console.log("[TypeTube Bridge] Could not notify tabs:", e);
            }
          } else {
            response = { error: "userId and settings are required" };
          }
          break;
        default:
          response = { error: "Unknown message type" };
      }
      window.postMessage({
        source: "typetube-extension",
        id: message.id,
        response
      }, "*");
    } catch (error) {
      console.error("[TypeTube Bridge] Error handling message:", error);
      window.postMessage({
        source: "typetube-extension",
        id: message.id,
        error: String(error)
      }, "*");
    }
  });
  window.postMessage({
    source: "typetube-extension",
    type: "EXTENSION_READY"
  }, "*");
  console.log("[TypeTube Bridge] Dashboard bridge initialized");
})();
