"use strict";
(() => {
  // extension/letterProgress.ts
  var LETTER_ORDER = [
    "e",
    "n",
    "i",
    "a",
    "r",
    "l",
    "t",
    "o",
    "s",
    "u",
    "d",
    "y",
    "c",
    "g",
    "h",
    "p",
    "m",
    "k",
    "b",
    "w",
    "f",
    "z",
    "v",
    "x",
    "q",
    "j"
  ];
  var INITIAL_LETTERS = LETTER_ORDER.slice(0, 8);
  var MASTERY_CRITERIA = {
    minConfidence: 0.45,
    // 45% confidence (lowered for adaptive scoring)
    maxErrorRate: 0.18,
    // 18% errors allowed (more forgiving)
    minSampleSize: 10
    // 10 samples per letter (was 15)
  };
  function initializeLetterProgress() {
    const letterStats = /* @__PURE__ */ new Map();
    for (const letter of LETTER_ORDER) {
      letterStats.set(letter, {
        letter,
        timesTyped: 0,
        errors: 0,
        totalTimeMs: 0,
        confidence: 0,
        isUnlocked: INITIAL_LETTERS.includes(letter),
        isMastered: false
      });
    }
    return {
      letterStats,
      unlockedLetters: new Set(INITIAL_LETTERS),
      currentEmphasisLetter: INITIAL_LETTERS[INITIAL_LETTERS.length - 1],
      lastUnlockTimestamp: Date.now(),
      globalStats: {
        totalCorrectKeystrokes: 0,
        totalTimeMs: 0
      },
      stageProgress: {
        capitalsIntroduced: false,
        capitalSuccessCount: 0,
        capitalErrorCount: 0,
        punctuationIntroduced: false,
        punctuationSuccessCount: 0,
        punctuationErrorCount: 0,
        sentencesUnlocked: false
      }
    };
  }
  async function loadLetterProgress() {
    try {
      const result = await chrome.storage.local.get(["letterProgress"]);
      if (result.letterProgress) {
        const data = result.letterProgress;
        const letterStats = /* @__PURE__ */ new Map();
        for (const [letter, stats] of Object.entries(data.letterStats || {})) {
          letterStats.set(letter, stats);
        }
        return {
          letterStats,
          unlockedLetters: new Set(data.unlockedLetters || INITIAL_LETTERS),
          currentEmphasisLetter: data.currentEmphasisLetter || null,
          lastUnlockTimestamp: data.lastUnlockTimestamp || Date.now(),
          globalStats: data.globalStats || {
            totalCorrectKeystrokes: 0,
            totalTimeMs: 0
          },
          stageProgress: data.stageProgress || {
            capitalsIntroduced: false,
            capitalSuccessCount: 0,
            capitalErrorCount: 0,
            punctuationIntroduced: false,
            punctuationSuccessCount: 0,
            punctuationErrorCount: 0,
            sentencesUnlocked: false
          }
        };
      }
    } catch (e) {
      console.error("[LetterProgress] Error loading from storage:", e);
    }
    return initializeLetterProgress();
  }
  async function saveLetterProgress(state2) {
    try {
      const data = {
        letterStats: Object.fromEntries(state2.letterStats),
        unlockedLetters: Array.from(state2.unlockedLetters),
        currentEmphasisLetter: state2.currentEmphasisLetter,
        lastUnlockTimestamp: state2.lastUnlockTimestamp,
        globalStats: state2.globalStats,
        stageProgress: state2.stageProgress
      };
      await chrome.storage.local.set({ letterProgress: data });
    } catch (e) {
      console.error("[LetterProgress] Error saving to storage:", e);
    }
  }
  function recordKeystroke(state2, letter, isCorrect, timeMs) {
    const stats = state2.letterStats.get(letter.toLowerCase());
    if (!stats) return;
    stats.timesTyped++;
    if (!isCorrect) {
      stats.errors++;
    }
    if (isCorrect && timeMs > 0) {
      stats.totalTimeMs += timeMs;
      state2.globalStats.totalCorrectKeystrokes++;
      state2.globalStats.totalTimeMs += timeMs;
    }
    updateConfidence(stats, state2.globalStats);
    checkMastery(stats);
  }
  var MIN_KEYSTROKES_FOR_ADAPTIVE = 50;
  var DEFAULT_BASELINE_MS = 500;
  function getBaselineSpeedMs(globalStats) {
    if (globalStats.totalCorrectKeystrokes < MIN_KEYSTROKES_FOR_ADAPTIVE) {
      return DEFAULT_BASELINE_MS;
    }
    return globalStats.totalTimeMs / globalStats.totalCorrectKeystrokes;
  }
  function updateConfidence(stats, globalStats) {
    if (stats.timesTyped === 0) {
      stats.confidence = 0;
      return;
    }
    const correctTypings = stats.timesTyped - stats.errors;
    const errorRate = stats.errors / stats.timesTyped;
    const avgTimeMs = correctTypings > 0 ? stats.totalTimeMs / correctTypings : 1e3;
    const baselineMs = getBaselineSpeedMs(globalStats);
    const speedRatio = avgTimeMs / baselineMs;
    const speedScore = Math.max(0, Math.min(1, 1.5 - speedRatio));
    const accuracyScore = Math.max(0, Math.min(1, 1 - errorRate * 5));
    const sampleScore = Math.min(1, stats.timesTyped / 50);
    stats.confidence = accuracyScore * 0.55 + speedScore * 0.25 + sampleScore * 0.2;
  }
  function checkMastery(stats) {
    if (stats.timesTyped < MASTERY_CRITERIA.minSampleSize) {
      stats.isMastered = false;
      return;
    }
    const errorRate = stats.errors / stats.timesTyped;
    stats.isMastered = stats.confidence >= MASTERY_CRITERIA.minConfidence && errorRate <= MASTERY_CRITERIA.maxErrorRate;
  }
  function getErrorRate(stats) {
    if (stats.timesTyped === 0) return 0;
    return stats.errors / stats.timesTyped;
  }
  function checkForNewLetterUnlock(state2) {
    const unlockedStats = Array.from(state2.letterStats.values()).filter((s) => s.isUnlocked);
    const masteredLetters = unlockedStats.filter((s) => s.isMastered);
    const notMastered = unlockedStats.filter((s) => !s.isMastered);
    if (notMastered.length > 0) {
      const notMasteredInfo = notMastered.map(
        (s) => `${s.letter.toUpperCase()}(${s.timesTyped}/${MASTERY_CRITERIA.minSampleSize}, ${Math.round(s.confidence * 100)}%conf, ${Math.round(s.errors / Math.max(1, s.timesTyped) * 100)}%err)`
      ).join(", ");
      console.log(`[LetterProgress] Not yet mastered: ${notMasteredInfo}`);
    }
    const allMastered = notMastered.length === 0;
    if (!allMastered) {
      return false;
    }
    const nextLetter = LETTER_ORDER.find((letter) => {
      const stats2 = state2.letterStats.get(letter);
      return stats2 && !stats2.isUnlocked;
    });
    if (!nextLetter) {
      console.log("[LetterProgress] All letters unlocked!");
      return false;
    }
    const stats = state2.letterStats.get(nextLetter);
    if (stats) {
      stats.isUnlocked = true;
      state2.unlockedLetters.add(nextLetter);
      state2.currentEmphasisLetter = nextLetter;
      state2.lastUnlockTimestamp = Date.now();
      console.log(`[LetterProgress] Unlocked new letter: ${nextLetter.toUpperCase()}`);
      return true;
    }
    return false;
  }
  function getStrugglingLetters(state2) {
    const struggling = [];
    for (const stats of state2.letterStats.values()) {
      if (!stats.isUnlocked) continue;
      if (stats.timesTyped < MASTERY_CRITERIA.minSampleSize) continue;
      const errorRate = getErrorRate(stats);
      if (stats.isMastered && stats.confidence < MASTERY_CRITERIA.minConfidence * 0.9 || errorRate > MASTERY_CRITERIA.maxErrorRate * 1.5) {
        struggling.push(stats.letter);
      }
    }
    return struggling;
  }
  function getEmphasisLetter(state2) {
    const struggling = getStrugglingLetters(state2);
    if (struggling.length > 0) {
      const worstLetter = struggling.reduce((worst, letter) => {
        const currentStats = state2.letterStats.get(letter);
        const worstStats = state2.letterStats.get(worst);
        if (!currentStats || !worstStats) return worst;
        return getErrorRate(currentStats) > getErrorRate(worstStats) ? letter : worst;
      });
      return worstLetter;
    }
    return state2.currentEmphasisLetter;
  }
  function getEstimatedWPM(globalStats) {
    if (globalStats.totalCorrectKeystrokes < MIN_KEYSTROKES_FOR_ADAPTIVE) {
      return null;
    }
    const avgMs = globalStats.totalTimeMs / globalStats.totalCorrectKeystrokes;
    return Math.round(6e4 / (avgMs * 5));
  }
  function getProgressSummary(state2) {
    const unlockedStats = Array.from(state2.letterStats.values()).filter((s) => s.isUnlocked);
    const masteredStats = unlockedStats.filter((s) => s.isMastered);
    const struggling = getStrugglingLetters(state2);
    return {
      unlockedCount: state2.unlockedLetters.size,
      masteredCount: masteredStats.length,
      strugglingCount: struggling.length,
      totalLetters: LETTER_ORDER.length,
      emphasisLetter: getEmphasisLetter(state2),
      estimatedWPM: getEstimatedWPM(state2.globalStats),
      baselineSpeedMs: getBaselineSpeedMs(state2.globalStats)
    };
  }
  var STAGE_MASTERY_CRITERIA = {
    // Minimum success count before advancing to next stage
    minCapitalSuccesses: 50,
    minPunctuationSuccesses: 30,
    // Maximum error rate to advance
    maxErrorRate: 0.2
  };
  function areAllLettersMastered(state2) {
    for (const letter of LETTER_ORDER) {
      const stats = state2.letterStats.get(letter);
      if (!stats || !stats.isMastered) {
        return false;
      }
    }
    return true;
  }
  function getCurrentStage(state2) {
    if (!areAllLettersMastered(state2)) {
      return "letters";
    }
    const sp = state2.stageProgress;
    const capitalErrorRate = sp.capitalSuccessCount > 0 ? sp.capitalErrorCount / (sp.capitalSuccessCount + sp.capitalErrorCount) : 1;
    const capitalsMastered = sp.capitalsIntroduced && sp.capitalSuccessCount >= STAGE_MASTERY_CRITERIA.minCapitalSuccesses && capitalErrorRate <= STAGE_MASTERY_CRITERIA.maxErrorRate;
    if (!capitalsMastered) {
      return "capitals";
    }
    const punctuationErrorRate = sp.punctuationSuccessCount > 0 ? sp.punctuationErrorCount / (sp.punctuationSuccessCount + sp.punctuationErrorCount) : 1;
    const punctuationMastered = sp.punctuationIntroduced && sp.punctuationSuccessCount >= STAGE_MASTERY_CRITERIA.minPunctuationSuccesses && punctuationErrorRate <= STAGE_MASTERY_CRITERIA.maxErrorRate;
    if (!punctuationMastered) {
      return "punctuation";
    }
    return "sentences";
  }
  function recordCapitalKeystroke(state2, isCorrect) {
    state2.stageProgress.capitalsIntroduced = true;
    if (isCorrect) {
      state2.stageProgress.capitalSuccessCount++;
    } else {
      state2.stageProgress.capitalErrorCount++;
    }
  }
  function recordPunctuationKeystroke(state2, isCorrect) {
    state2.stageProgress.punctuationIntroduced = true;
    if (isCorrect) {
      state2.stageProgress.punctuationSuccessCount++;
    } else {
      state2.stageProgress.punctuationErrorCount++;
    }
  }
  function isCapitalLetter(char) {
    return /^[A-Z]$/.test(char);
  }
  function isPunctuation(char) {
    return /^[.,!?]$/.test(char);
  }

  // extension/words/wordList.ts
  var RAW_WORDS = [
    // Stage 1: E, N, I, A (core vowels + common consonants)
    "in",
    "an",
    "ai",
    "na",
    // Stage 2: E, N, I, A, R
    "ear",
    "era",
    "ran",
    "rain",
    "air",
    "are",
    "ran",
    "near",
    // Stage 3: E, N, I, A, R, L
    "line",
    "lane",
    "lean",
    "nail",
    "lane",
    "earl",
    "real",
    "liar",
    "liner",
    "alien",
    // Stage 4: E, N, I, A, R, L, T
    "ten",
    "net",
    "tin",
    "tan",
    "ant",
    "art",
    "eat",
    "tea",
    "rat",
    "tar",
    "tie",
    "ate",
    "late",
    "tear",
    "tale",
    "neat",
    "tree",
    "tile",
    "train",
    "trail",
    "learn",
    "later",
    "rate",
    "tail",
    "enter",
    "rent",
    "tire",
    "tent",
    "treat",
    "trial",
    "talent",
    "initial",
    "retreat",
    // Stage 5: E, N, I, A, R, L, T, O
    "one",
    "ton",
    "not",
    "into",
    "to",
    "too",
    "on",
    "or",
    "for",
    "torn",
    "tool",
    "near",
    "note",
    "tone",
    "tore",
    "oral",
    "iron",
    "loot",
    "noon",
    "onto",
    "lion",
    "ortion",
    "rotten",
    "lotion",
    "learn",
    "relation",
    "rational",
    "national",
    "location",
    // Stage 6: + S (adds plurals and many common words)
    "is",
    "as",
    "so",
    "sea",
    "sit",
    "set",
    "sore",
    "seal",
    "seat",
    "salt",
    "seen",
    "soon",
    "sail",
    "star",
    "stare",
    "store",
    "stone",
    "stolen",
    "sister",
    "lesson",
    "station",
    "season",
    "reason",
    "stories",
    "eastern",
    "tension",
    "sleet",
    "slate",
    "snail",
    "stain",
    "serial",
    "steal",
    // Stage 7: + U
    "us",
    "use",
    "user",
    "sure",
    "turn",
    "tune",
    "unit",
    "until",
    "us",
    "under",
    "rulers",
    "utils",
    "useful",
    "unusual",
    "natural",
    // Stage 8: + D
    "do",
    "ad",
    "id",
    "did",
    "dad",
    "and",
    "end",
    "den",
    "red",
    "led",
    "led",
    "lad",
    "rid",
    "rod",
    "nod",
    "odd",
    "add",
    "date",
    "deal",
    "dead",
    "dear",
    "dire",
    "done",
    "dine",
    "ride",
    "side",
    "said",
    "land",
    "send",
    "sand",
    "lead",
    "read",
    "road",
    "load",
    "node",
    "seed",
    "need",
    "used",
    "tied",
    "tried",
    "trade",
    "aside",
    "ideal",
    "radio",
    "solid",
    "dated",
    "detail",
    "dread",
    "stand",
    "island",
    "under",
    "dollar",
    "second",
    "record",
    "random",
    "modern",
    "studied",
    // Stage 9: + Y
    "yes",
    "yet",
    "you",
    "yell",
    "year",
    "your",
    "yeast",
    "yesterday",
    "ray",
    "say",
    "day",
    "lay",
    "may",
    "any",
    "try",
    "dry",
    "sly",
    "toys",
    "yard",
    "yarn",
    "yanks",
    "yells",
    "yearly",
    "youth",
    "yonder",
    "yeti",
    "yodel",
    "delay",
    "relay",
    "years",
    "ready",
    "steady",
    "tardy",
    "sandy",
    "rainy",
    "study",
    "dusty",
    "rusty",
    "nasty",
    "tasty",
    "style",
    // Stage 10: + C
    "can",
    "cat",
    "car",
    "cut",
    "cone",
    "cute",
    "care",
    "case",
    "cost",
    "code",
    "cold",
    "call",
    "coal",
    "cent",
    "cite",
    "core",
    "corn",
    "Carol",
    "clear",
    "clean",
    "cloud",
    "class",
    "close",
    "castle",
    "cradle",
    "course",
    "circle",
    "cancel",
    "circus",
    "ocean",
    "local",
    "recall",
    "record",
    "recent",
    "society",
    "social",
    "actual",
    "council",
    // Stage 11: + G
    "go",
    "got",
    "get",
    "leg",
    "gel",
    "tag",
    "gas",
    "gal",
    "rag",
    "sag",
    "gag",
    "gate",
    "rage",
    "sage",
    "stage",
    "giant",
    "grace",
    "grand",
    "grade",
    "great",
    "green",
    "greet",
    "angel",
    "tiger",
    "magic",
    "legal",
    "signal",
    "region",
    "garden",
    "garage",
    "goal",
    "general",
    "generate",
    // Stage 12: + H
    "hi",
    "he",
    "hat",
    "hot",
    "hit",
    "her",
    "has",
    "had",
    "hut",
    "hug",
    "hen",
    "hair",
    "hand",
    "hard",
    "hear",
    "heat",
    "heal",
    "hole",
    "hold",
    "heart",
    "horse",
    "house",
    "hours",
    "heads",
    "ahead",
    "health",
    "honest",
    "handle",
    "happen",
    "higher",
    "highly",
    "huge",
    "hurry",
    "hotel",
    "harsh",
    "chart",
    "chair",
    "chain",
    "cheer",
    "chess",
    "chose",
    "those",
    "others",
    "rather",
    "either",
    "gather",
    "whether",
    "athlete",
    "thunder",
    "thought",
    // Stage 13: + P
    "up",
    "pen",
    "pin",
    "pan",
    "pun",
    "put",
    "pat",
    "pet",
    "pit",
    "pot",
    "pie",
    "pea",
    "ape",
    "sip",
    "tap",
    "tip",
    "top",
    "pop",
    "lap",
    "nap",
    "page",
    "pain",
    "pair",
    "pale",
    "part",
    "pass",
    "past",
    "path",
    "pear",
    "piece",
    "place",
    "plane",
    "plant",
    "plate",
    "point",
    "print",
    "price",
    "pride",
    "prize",
    "paper",
    "parent",
    "peace",
    "people",
    "person",
    "phrase",
    "planet",
    "plenty",
    "poetry",
    "polish",
    "popular",
    "practice",
    "prepare",
    "principal",
    "product",
    "program",
    "project",
    "protect",
    "purpose",
    // Stage 14: + M
    "am",
    "me",
    "my",
    "may",
    "man",
    "men",
    "mat",
    "met",
    "mom",
    "mop",
    "mad",
    "map",
    "mix",
    "mud",
    "arm",
    "aim",
    "ham",
    "him",
    "hum",
    "ram",
    "rim",
    "rum",
    "sum",
    "made",
    "make",
    "male",
    "many",
    "mark",
    "meal",
    "mean",
    "meet",
    "mile",
    "mind",
    "mine",
    "more",
    "most",
    "move",
    "must",
    "music",
    "major",
    "march",
    "match",
    "maybe",
    "metal",
    "meter",
    "might",
    "minor",
    "minus",
    "model",
    "modern",
    "money",
    "month",
    "moral",
    "motor",
    "mouse",
    "mouth",
    "movie",
    "machine",
    "material",
    "measure",
    "medical",
    "medium",
    "member",
    "memory",
    "mental",
    "method",
    "middle",
    "minute",
    "moment",
    "morning",
    "mother",
    "motion",
    "mountain",
    "movement",
    "museum",
    // Stage 15: + K
    "ok",
    "key",
    "kid",
    "kit",
    "ink",
    "ask",
    "oak",
    "ark",
    "irk",
    "elk",
    "kale",
    "keen",
    "keep",
    "kept",
    "kick",
    "kind",
    "king",
    "kiss",
    "kite",
    "knee",
    "knew",
    "know",
    "take",
    "lake",
    "like",
    "make",
    "mike",
    "rake",
    "sake",
    "talk",
    "walk",
    "work",
    "dark",
    "park",
    "mark",
    "mask",
    "task",
    "desk",
    "disk",
    "risk",
    "token",
    "track",
    "trick",
    "truck",
    "kayak",
    "kitchen",
    "kingdom",
    "knowledge",
    // Stage 16: + B
    "be",
    "by",
    "but",
    "bus",
    "boy",
    "buy",
    "bet",
    "big",
    "bit",
    "bad",
    "bag",
    "ban",
    "bar",
    "bat",
    "bed",
    "bee",
    "beg",
    "bin",
    "boa",
    "bob",
    "bug",
    "bun",
    "baby",
    "back",
    "bake",
    "ball",
    "band",
    "base",
    "bath",
    "bear",
    "beat",
    "been",
    "best",
    "bill",
    "bird",
    "blue",
    "boat",
    "bone",
    "book",
    "born",
    "both",
    "bread",
    "break",
    "bring",
    "build",
    "table",
    "object",
    "number",
    "public",
    "describe",
    "balance",
    "better",
    "bottle",
    "bottom",
    "brain",
    "branch",
    "brother",
    "brought",
    // Stage 17: + W
    "we",
    "was",
    "wet",
    "won",
    "wow",
    "way",
    "war",
    "who",
    "why",
    "win",
    "wit",
    "wag",
    "wig",
    "wade",
    "wage",
    "wait",
    "wake",
    "walk",
    "wall",
    "want",
    "warm",
    "warn",
    "wash",
    "wave",
    "weak",
    "wear",
    "west",
    "what",
    "when",
    "wide",
    "wife",
    "wild",
    "will",
    "wind",
    "wine",
    "wing",
    "wire",
    "wise",
    "wish",
    "with",
    "woke",
    "wolf",
    "wood",
    "word",
    "work",
    "world",
    "worry",
    "would",
    "write",
    "wrote",
    "water",
    "watch",
    "wealth",
    "weather",
    "weight",
    "welcome",
    "window",
    "winter",
    "wonder",
    "woman",
    "women",
    // Stage 18: + F
    "if",
    "of",
    "off",
    "for",
    "far",
    "fat",
    "fan",
    "fun",
    "fit",
    "fir",
    "fee",
    "few",
    "fog",
    "fin",
    "fix",
    "fad",
    "face",
    "fact",
    "fail",
    "fair",
    "fall",
    "fame",
    "farm",
    "fast",
    "fate",
    "fear",
    "feel",
    "feet",
    "fell",
    "felt",
    "file",
    "fill",
    "film",
    "find",
    "fine",
    "fire",
    "firm",
    "fish",
    "fist",
    "five",
    "flat",
    "flew",
    "flow",
    "folk",
    "food",
    "foot",
    "force",
    "form",
    "fort",
    "four",
    "free",
    "from",
    "front",
    "fruit",
    "full",
    "final",
    "finger",
    "finish",
    "follow",
    "forest",
    "forget",
    "formal",
    "forward",
    "future",
    "figure",
    "father",
    "family",
    "famous",
    "favorite",
    // Stage 19: + Z
    "zen",
    "zap",
    "zip",
    "zoo",
    "zero",
    "zone",
    "zeal",
    "size",
    "quiz",
    "fizz",
    "fuzz",
    "jazz",
    "lazy",
    "haze",
    "maze",
    "doze",
    "froze",
    "graze",
    // Stage 20: + V
    "van",
    "vet",
    "vat",
    "via",
    "vie",
    "vow",
    "vile",
    "vine",
    "visa",
    "visit",
    "vote",
    "voice",
    "value",
    "valid",
    "valve",
    "velvet",
    "vendor",
    "verbal",
    "very",
    "video",
    "view",
    "villa",
    "vital",
    "vocal",
    "volume",
    "vowel",
    "travel",
    "over",
    "ever",
    "even",
    "every",
    "river",
    "cover",
    "lover",
    "never",
    "seven",
    "fever",
    "clever",
    "silver",
    "deliver",
    "develop",
    // Stage 21: + X
    "ox",
    "ax",
    "fox",
    "fix",
    "mix",
    "six",
    "wax",
    "tax",
    "box",
    "next",
    "exit",
    "exam",
    "exact",
    "example",
    "expert",
    "explain",
    "explore",
    // Stage 22: + Q (always with U)
    "quit",
    "quiz",
    "queen",
    "quest",
    "quick",
    "quiet",
    "quite",
    "quote",
    "equal",
    "equip",
    "require",
    "question",
    "quality",
    "quantity",
    // Stage 23: + J
    "jar",
    "jam",
    "jet",
    "job",
    "joy",
    "jog",
    "join",
    "joke",
    "jump",
    "just",
    "judge",
    "juice",
    "junior",
    "jungle",
    "journey",
    "journal",
    "object",
    "subject",
    "project",
    "major",
    "enjoy"
  ];
  function processWords(rawWords) {
    const words = [];
    const seen = /* @__PURE__ */ new Set();
    for (const text of rawWords) {
      const normalized = text.toLowerCase().trim();
      if (!normalized || seen.has(normalized) || !/^[a-z]+$/.test(normalized)) {
        continue;
      }
      seen.add(normalized);
      const letters = new Set(normalized.split(""));
      words.push({
        text: normalized,
        letters,
        length: normalized.length
      });
    }
    return words;
  }
  var WORD_DATABASE = processWords(RAW_WORDS);
  function getWordsWithLetters(allowedLetters) {
    return WORD_DATABASE.filter((word) => {
      for (const letter of word.letters) {
        if (!allowedLetters.has(letter)) {
          return false;
        }
      }
      return true;
    });
  }
  function getWordsContaining(allowedLetters, requiredLetter) {
    return WORD_DATABASE.filter((word) => {
      if (!word.letters.has(requiredLetter)) {
        return false;
      }
      for (const letter of word.letters) {
        if (!allowedLetters.has(letter)) {
          return false;
        }
      }
      return true;
    });
  }
  function getWordsByLength(words, minLength, maxLength) {
    return words.filter((word) => word.length >= minLength && word.length <= maxLength);
  }

  // extension/words/sentenceList.ts
  var RAW_SENTENCES = [
    // === CAPITALS DIFFICULTY (just capital at start, period at end) ===
    // Simple facts
    { text: "The sun is very hot.", category: "simple_fact", difficulty: "capitals" },
    { text: "Dogs like to run.", category: "simple_fact", difficulty: "capitals" },
    { text: "Cats can jump high.", category: "simple_fact", difficulty: "capitals" },
    { text: "Fish live in water.", category: "simple_fact", difficulty: "capitals" },
    { text: "Birds can fly.", category: "simple_fact", difficulty: "capitals" },
    { text: "The sky is blue.", category: "simple_fact", difficulty: "capitals" },
    { text: "Trees need water.", category: "simple_fact", difficulty: "capitals" },
    { text: "Bees make honey.", category: "simple_fact", difficulty: "capitals" },
    { text: "Snow is cold.", category: "simple_fact", difficulty: "capitals" },
    { text: "Ice is frozen water.", category: "simple_fact", difficulty: "capitals" },
    { text: "Plants need sunlight.", category: "simple_fact", difficulty: "capitals" },
    { text: "The moon shines at night.", category: "simple_fact", difficulty: "capitals" },
    { text: "Apples grow on trees.", category: "simple_fact", difficulty: "capitals" },
    { text: "Frogs can hop.", category: "simple_fact", difficulty: "capitals" },
    { text: "Rain comes from clouds.", category: "simple_fact", difficulty: "capitals" },
    // Fun phrases
    { text: "I like pizza.", category: "fun_phrase", difficulty: "capitals" },
    { text: "My dog is fluffy.", category: "fun_phrase", difficulty: "capitals" },
    { text: "Today is a good day.", category: "fun_phrase", difficulty: "capitals" },
    { text: "I love ice cream.", category: "fun_phrase", difficulty: "capitals" },
    { text: "Books are fun to read.", category: "fun_phrase", difficulty: "capitals" },
    { text: "Summer is my favorite.", category: "fun_phrase", difficulty: "capitals" },
    { text: "I have a pet fish.", category: "fun_phrase", difficulty: "capitals" },
    { text: "My friend is kind.", category: "fun_phrase", difficulty: "capitals" },
    { text: "I can ride a bike.", category: "fun_phrase", difficulty: "capitals" },
    { text: "Stars twinkle at night.", category: "fun_phrase", difficulty: "capitals" },
    { text: "The rainbow is pretty.", category: "fun_phrase", difficulty: "capitals" },
    { text: "I like to draw.", category: "fun_phrase", difficulty: "capitals" },
    { text: "Music makes me happy.", category: "fun_phrase", difficulty: "capitals" },
    { text: "I can count to ten.", category: "fun_phrase", difficulty: "capitals" },
    { text: "My room is clean.", category: "fun_phrase", difficulty: "capitals" },
    // Tongue twisters (simplified for capitals stage)
    { text: "She sells sea shells.", category: "tongue_twister", difficulty: "capitals" },
    { text: "Red lorry yellow lorry.", category: "tongue_twister", difficulty: "capitals" },
    { text: "Toy boat toy boat.", category: "tongue_twister", difficulty: "capitals" },
    { text: "A big bug bit a bear.", category: "tongue_twister", difficulty: "capitals" },
    { text: "Six slippery snails.", category: "tongue_twister", difficulty: "capitals" },
    // === PUNCTUATION DIFFICULTY (adds commas, ! and ?) ===
    // Simple facts with commas
    { text: "The dog ran, jumped, and played.", category: "simple_fact", difficulty: "punctuation" },
    { text: "I like apples, oranges, and grapes.", category: "simple_fact", difficulty: "punctuation" },
    { text: "We have cats, dogs, and birds.", category: "simple_fact", difficulty: "punctuation" },
    { text: "Red, blue, and green are colors.", category: "simple_fact", difficulty: "punctuation" },
    { text: "Lions, tigers, and bears are animals.", category: "simple_fact", difficulty: "punctuation" },
    // Exclamations
    { text: "What a great day!", category: "fun_phrase", difficulty: "punctuation" },
    { text: "I love this game!", category: "fun_phrase", difficulty: "punctuation" },
    { text: "That was amazing!", category: "fun_phrase", difficulty: "punctuation" },
    { text: "You did it!", category: "fun_phrase", difficulty: "punctuation" },
    { text: "Hooray for ice cream!", category: "fun_phrase", difficulty: "punctuation" },
    { text: "What a big dog!", category: "fun_phrase", difficulty: "punctuation" },
    { text: "I can do it!", category: "fun_phrase", difficulty: "punctuation" },
    { text: "That is so cool!", category: "fun_phrase", difficulty: "punctuation" },
    { text: "Go team go!", category: "fun_phrase", difficulty: "punctuation" },
    { text: "Happy birthday!", category: "fun_phrase", difficulty: "punctuation" },
    // Questions
    { text: "Do you like pizza?", category: "question", difficulty: "punctuation" },
    { text: "What is your name?", category: "question", difficulty: "punctuation" },
    { text: "Where do you live?", category: "question", difficulty: "punctuation" },
    { text: "Can you help me?", category: "question", difficulty: "punctuation" },
    { text: "How are you today?", category: "question", difficulty: "punctuation" },
    { text: "What time is it?", category: "question", difficulty: "punctuation" },
    { text: "Do you have a pet?", category: "question", difficulty: "punctuation" },
    { text: "Is it raining outside?", category: "question", difficulty: "punctuation" },
    { text: "What is your favorite color?", category: "question", difficulty: "punctuation" },
    { text: "Are you ready to go?", category: "question", difficulty: "punctuation" },
    // Mixed punctuation
    { text: "Wow, that is amazing!", category: "fun_phrase", difficulty: "punctuation" },
    { text: "Hey, want to play?", category: "question", difficulty: "punctuation" },
    { text: "Look, a rainbow!", category: "fun_phrase", difficulty: "punctuation" },
    { text: "Yes, I can do that!", category: "fun_phrase", difficulty: "punctuation" },
    { text: "Wait, is that a bird?", category: "question", difficulty: "punctuation" },
    // === SENTENCES DIFFICULTY (full variety) ===
    // Complex facts
    { text: "The quick brown fox jumps over the lazy dog.", category: "simple_fact", difficulty: "sentences" },
    { text: "Dolphins are smart, friendly, and playful animals.", category: "simple_fact", difficulty: "sentences" },
    { text: "The Earth orbits around the sun once each year.", category: "simple_fact", difficulty: "sentences" },
    { text: "Rainbows appear when sunlight passes through rain.", category: "simple_fact", difficulty: "sentences" },
    { text: "Butterflies start as tiny caterpillars.", category: "simple_fact", difficulty: "sentences" },
    { text: "Elephants are the largest land animals on Earth.", category: "simple_fact", difficulty: "sentences" },
    { text: "Penguins cannot fly, but they swim very well.", category: "simple_fact", difficulty: "sentences" },
    { text: "The ocean is home to many amazing creatures.", category: "simple_fact", difficulty: "sentences" },
    // Engaging phrases
    { text: "I believe I can do anything I set my mind to!", category: "fun_phrase", difficulty: "sentences" },
    { text: "Practice makes perfect, so keep trying!", category: "fun_phrase", difficulty: "sentences" },
    { text: "Every day is a new adventure waiting to happen.", category: "fun_phrase", difficulty: "sentences" },
    { text: "With friends by your side, anything is possible!", category: "fun_phrase", difficulty: "sentences" },
    { text: "Dream big, work hard, and never give up!", category: "fun_phrase", difficulty: "sentences" },
    { text: "Kindness is free, so share it everywhere!", category: "fun_phrase", difficulty: "sentences" },
    { text: "Be brave, be bold, be you!", category: "fun_phrase", difficulty: "sentences" },
    { text: "The best time to start is right now!", category: "fun_phrase", difficulty: "sentences" },
    // Tongue twisters
    { text: "Peter Piper picked a peck of pickled peppers.", category: "tongue_twister", difficulty: "sentences" },
    { text: "How much wood would a woodchuck chuck?", category: "tongue_twister", difficulty: "sentences" },
    { text: "Fuzzy Wuzzy was a bear, Fuzzy Wuzzy had no hair.", category: "tongue_twister", difficulty: "sentences" },
    { text: "Betty Botter bought some butter for her batter.", category: "tongue_twister", difficulty: "sentences" },
    { text: "Sally sells seashells by the seashore.", category: "tongue_twister", difficulty: "sentences" },
    { text: "Six sticky skeletons sat on six slim sticks.", category: "tongue_twister", difficulty: "sentences" },
    { text: "Rubber baby buggy bumpers bounce.", category: "tongue_twister", difficulty: "sentences" },
    { text: "A proper copper coffee pot is hard to find.", category: "tongue_twister", difficulty: "sentences" },
    // Complex questions
    { text: "Have you ever seen a shooting star at night?", category: "question", difficulty: "sentences" },
    { text: "What would you do if you could fly like a bird?", category: "question", difficulty: "sentences" },
    { text: "If you could visit any place, where would you go?", category: "question", difficulty: "sentences" },
    { text: "What is your favorite thing about summer vacation?", category: "question", difficulty: "sentences" },
    { text: "Can you imagine a world without any colors?", category: "question", difficulty: "sentences" },
    { text: "If animals could talk, what would they say?", category: "question", difficulty: "sentences" },
    { text: "What makes you smile on a rainy day?", category: "question", difficulty: "sentences" },
    { text: "Do you think robots will be our friends someday?", category: "question", difficulty: "sentences" },
    // Mixed complex
    { text: "Wow, did you see that amazing rainbow!", category: "fun_phrase", difficulty: "sentences" },
    { text: "Look up at the stars; they are so beautiful tonight!", category: "fun_phrase", difficulty: "sentences" },
    { text: "Ready, set, go!", category: "fun_phrase", difficulty: "sentences" },
    { text: "First, gather the materials; then, start building.", category: "simple_fact", difficulty: "sentences" },
    { text: "Apples, bananas, cherries, and dates are all fruits.", category: "simple_fact", difficulty: "sentences" }
  ];
  function processSentences(raw) {
    return raw.map((item) => ({
      text: item.text,
      category: item.category,
      difficulty: item.difficulty,
      wordCount: item.text.split(/\s+/).length
    }));
  }
  var SENTENCE_DATABASE = processSentences(RAW_SENTENCES);
  function getSentencesForStage(stage) {
    switch (stage) {
      case "letters":
        return [];
      case "capitals":
        return SENTENCE_DATABASE.filter((s) => s.difficulty === "capitals");
      case "punctuation":
        return SENTENCE_DATABASE.filter(
          (s) => s.difficulty === "capitals" || s.difficulty === "punctuation"
        );
      case "sentences":
        return SENTENCE_DATABASE;
    }
  }
  function selectRandomSentence(stage, recentSentences2, maxWordCount) {
    let candidates = getSentencesForStage(stage);
    if (maxWordCount !== void 0) {
      candidates = candidates.filter((s) => s.wordCount <= maxWordCount);
    }
    const available = candidates.filter((s) => !recentSentences2.has(s.text));
    if (available.length === 0) {
      if (candidates.length === 0) return null;
      return candidates[Math.floor(Math.random() * candidates.length)];
    }
    return available[Math.floor(Math.random() * available.length)];
  }

  // extension/captionExtractor.ts
  var CACHE_DURATION_MS = 60 * 60 * 1e3;
  var MIN_WORD_LENGTH = 2;
  var MAX_WORD_LENGTH = 12;
  var PREFERRED_LANGUAGES = ["en", "en-US", "en-GB", "en-AU", "en-CA"];
  function getVideoIdFromUrl() {
    const url = new URL(window.location.href);
    return url.searchParams.get("v");
  }
  async function extractCaptions(videoId) {
    console.log(`[CaptionExtractor] Extracting captions for video: ${videoId}`);
    const cached = await getCachedCaptions(videoId);
    if (cached) {
      console.log(`[CaptionExtractor] Using cached captions (${cached.words.length} words from ${cached.source})`);
      return cached.words;
    }
    let words = [];
    let source = "metadata";
    const captionTrack = await getCaptionTrackFromPlayerResponse();
    if (captionTrack) {
      words = await fetchAndParseTimedText(captionTrack);
      if (words.length > 0) {
        source = "timedtext";
        console.log(`[CaptionExtractor] Got ${words.length} words from timedtext API`);
      }
    }
    if (words.length === 0) {
      words = parsePlayerResponseCaptions();
      if (words.length > 0) {
        source = "playerResponse";
        console.log(`[CaptionExtractor] Got ${words.length} words from player response`);
      }
    }
    if (words.length === 0) {
      words = extractFromMetadata();
      source = "metadata";
      console.log(`[CaptionExtractor] Using ${words.length} words from metadata (fallback)`);
    }
    if (words.length > 0) {
      await cacheCaptions(videoId, words, source);
    }
    return words;
  }
  async function getCaptionTrackFromPlayerResponse() {
    try {
      const playerResponse = await getPlayerResponse();
      if (!playerResponse) return null;
      const captions = playerResponse?.captions?.playerCaptionsTracklistRenderer;
      if (!captions?.captionTracks || captions.captionTracks.length === 0) {
        console.log("[CaptionExtractor] No caption tracks found");
        return null;
      }
      const tracks = captions.captionTracks;
      let track = tracks.find(
        (t) => PREFERRED_LANGUAGES.some(
          (lang) => t.languageCode?.toLowerCase().startsWith(lang.toLowerCase()) || t.vssId?.includes(`.${lang}`)
        )
      );
      if (!track) {
        track = tracks[0];
        console.log(`[CaptionExtractor] No English track, using: ${track.languageCode || track.vssId}`);
      } else {
        console.log(`[CaptionExtractor] Found caption track: ${track.languageCode || track.vssId}`);
      }
      return track.baseUrl || null;
    } catch (e) {
      console.error("[CaptionExtractor] Error getting caption track:", e);
      return null;
    }
  }
  async function getPlayerResponse() {
    try {
      const scripts = document.querySelectorAll("script");
      for (let i = 0; i < scripts.length; i++) {
        const script = scripts[i];
        const text = script.textContent || "";
        const playerResponseMatch = text.match(/var\s+ytInitialPlayerResponse\s*=\s*(\{[\s\S]+?\});/);
        if (playerResponseMatch) {
          try {
            return JSON.parse(playerResponseMatch[1]);
          } catch (e) {
            const jsonStr = playerResponseMatch[1];
            const cleanJson = findValidJson(jsonStr);
            if (cleanJson) {
              return JSON.parse(cleanJson);
            }
          }
        }
        const initialDataMatch = text.match(/var\s+ytInitialData\s*=\s*(\{[\s\S]+?\});/);
        if (initialDataMatch) {
          try {
            return JSON.parse(initialDataMatch[1]);
          } catch (e) {
          }
        }
      }
    } catch (e) {
      console.error("[CaptionExtractor] Error parsing player response:", e);
    }
    return null;
  }
  function findValidJson(str) {
    let depth = 0;
    let inString = false;
    let escapeNext = false;
    for (let i = 0; i < str.length; i++) {
      const char = str[i];
      if (escapeNext) {
        escapeNext = false;
        continue;
      }
      if (char === "\\" && inString) {
        escapeNext = true;
        continue;
      }
      if (char === '"') {
        inString = !inString;
        continue;
      }
      if (inString) continue;
      if (char === "{") {
        depth++;
      } else if (char === "}") {
        depth--;
        if (depth === 0) {
          return str.substring(0, i + 1);
        }
      }
    }
    return null;
  }
  async function fetchAndParseTimedText(captionUrl) {
    try {
      const url = new URL(captionUrl);
      url.searchParams.set("fmt", "srv3");
      const response = await fetch(url.toString());
      if (!response.ok) {
        console.error(`[CaptionExtractor] Caption fetch failed: ${response.status}`);
        return [];
      }
      const text = await response.text();
      return parseCaptionXml(text);
    } catch (e) {
      console.error("[CaptionExtractor] Error fetching captions:", e);
      return [];
    }
  }
  function parseCaptionXml(xml) {
    const words = [];
    try {
      const parser = new DOMParser();
      const doc = parser.parseFromString(xml, "text/xml");
      const textElements = doc.querySelectorAll("text, p, s");
      textElements.forEach((element) => {
        const content = element.textContent || "";
        const extracted = extractWordsFromText(content);
        words.push(...extracted);
      });
      if (words.length === 0) {
        const rawText = doc.documentElement?.textContent || xml;
        words.push(...extractWordsFromText(rawText));
      }
    } catch (e) {
      console.error("[CaptionExtractor] Error parsing caption XML:", e);
      words.push(...extractWordsFromText(xml));
    }
    return deduplicateAndClean(words);
  }
  function parsePlayerResponseCaptions() {
    const captionContainers = document.querySelectorAll(
      '.ytp-caption-segment, .captions-text, [class*="caption"]'
    );
    const words = [];
    captionContainers.forEach((container) => {
      const text = container.textContent || "";
      words.push(...extractWordsFromText(text));
    });
    return deduplicateAndClean(words);
  }
  function extractFromMetadata() {
    const words = [];
    const titleElement = document.querySelector(
      'h1.ytd-video-primary-info-renderer yt-formatted-string, h1.ytd-watch-metadata yt-formatted-string, #title h1, meta[property="og:title"]'
    );
    if (titleElement) {
      const title = titleElement.textContent || titleElement.content || "";
      words.push(...extractWordsFromText(title));
    }
    const descElement = document.querySelector(
      '#description-text, ytd-text-inline-expander, meta[property="og:description"]'
    );
    if (descElement) {
      const desc = (descElement.textContent || descElement.content || "").substring(0, 500);
      words.push(...extractWordsFromText(desc));
    }
    const channelElement = document.querySelector(
      "#channel-name a, ytd-channel-name a, .ytd-video-owner-renderer #channel-name"
    );
    if (channelElement) {
      const channel = channelElement.textContent || "";
      words.push(...extractWordsFromText(channel));
    }
    return deduplicateAndClean(words);
  }
  function extractWordsFromText(text) {
    const decoded = decodeHtmlEntities(text);
    const rawWords = decoded.toLowerCase().split(/[\s\n\r\t.,!?;:'"()\[\]{}<>\/\\|@#$%^&*+=`~]+/).filter(Boolean);
    return rawWords.map((word) => word.replace(/[^a-z]/g, "")).filter(
      (word) => word.length >= MIN_WORD_LENGTH && word.length <= MAX_WORD_LENGTH && /^[a-z]+$/.test(word)
      // Only alphabetic
    );
  }
  function decodeHtmlEntities(text) {
    const entities = {
      "&amp;": "&",
      "&lt;": "<",
      "&gt;": ">",
      "&quot;": '"',
      "&#39;": "'",
      "&apos;": "'",
      "&nbsp;": " ",
      "&#x27;": "'",
      "&#x2F;": "/"
    };
    let result = text;
    for (const [entity, char] of Object.entries(entities)) {
      result = result.replace(new RegExp(entity, "gi"), char);
    }
    result = result.replace(
      /&#(\d+);/g,
      (_, code) => String.fromCharCode(parseInt(code, 10))
    );
    result = result.replace(
      /&#x([0-9a-f]+);/gi,
      (_, code) => String.fromCharCode(parseInt(code, 16))
    );
    return result;
  }
  function deduplicateAndClean(words) {
    const seen = /* @__PURE__ */ new Set();
    const result = [];
    for (const word of words) {
      if (!seen.has(word)) {
        seen.add(word);
        result.push(word);
      }
    }
    return result;
  }
  function filterByUnlockedLetters(words, unlockedLetters) {
    return words.filter((word) => {
      for (const char of word) {
        if (!unlockedLetters.has(char)) {
          return false;
        }
      }
      return true;
    }).map((word) => ({
      text: word,
      letters: new Set(word.split(""))
    }));
  }
  function filterByContainingLetter(words, letter) {
    return words.filter((word) => word.letters.has(letter));
  }
  async function getCachedCaptions(videoId) {
    try {
      const key = `captions_${videoId}`;
      const result = await chrome.storage.local.get([key]);
      if (result[key]) {
        const cache = result[key];
        if (Date.now() - cache.extractedAt < CACHE_DURATION_MS) {
          return cache;
        }
      }
    } catch (e) {
      console.error("[CaptionExtractor] Error reading cache:", e);
    }
    return null;
  }
  async function cacheCaptions(videoId, words, source) {
    try {
      const key = `captions_${videoId}`;
      const cache = {
        videoId,
        words,
        extractedAt: Date.now(),
        source
      };
      await chrome.storage.local.set({ [key]: cache });
      console.log(`[CaptionExtractor] Cached ${words.length} words for video ${videoId}`);
      await cleanupOldCaches();
    } catch (e) {
      console.error("[CaptionExtractor] Error caching captions:", e);
    }
  }
  async function cleanupOldCaches() {
    try {
      const result = await chrome.storage.local.get(null);
      const captionKeys = Object.keys(result).filter((k) => k.startsWith("captions_"));
      if (captionKeys.length > 20) {
        const caches = captionKeys.map((key) => ({ key, cache: result[key] })).sort((a, b) => b.cache.extractedAt - a.cache.extractedAt);
        const toRemove = caches.slice(20).map((c) => c.key);
        await chrome.storage.local.remove(toRemove);
        console.log(`[CaptionExtractor] Cleaned up ${toRemove.length} old caches`);
      }
    } catch (e) {
      console.error("[CaptionExtractor] Error cleaning caches:", e);
    }
  }

  // extension/content.ts
  function getAccuracyMultiplier(accuracy) {
    if (accuracy >= 98) return 1.5;
    if (accuracy >= 95) return 1.2;
    if (accuracy >= 90) return 1;
    if (accuracy >= 80) return 0.5;
    return 0.25;
  }
  function getStreakMultiplier(streak) {
    if (streak >= 50) return 2.5;
    if (streak >= 20) return 2;
    if (streak >= 10) return 1.5;
    if (streak >= 5) return 1.2;
    return 1;
  }
  function calculateErrorPenalty(baseFillPerChar) {
    return baseFillPerChar * 3;
  }
  function calculateFillAmount(baseFillPerChar, accuracy, streak) {
    const accuracyMultiplier = getAccuracyMultiplier(accuracy);
    const streakMultiplier = getStreakMultiplier(streak);
    return baseFillPerChar * accuracyMultiplier * streakMultiplier;
  }
  function calculateDrainRate(estimatedWPM, targetTypingRatio = 0.35, charactersPerWord = 5) {
    const charsPerMinute = estimatedWPM * charactersPerWord;
    const charsPerSecond = charsPerMinute / 60;
    return charsPerSecond * (targetTypingRatio / (1 - targetTypingRatio));
  }
  function calculateBaseFillPerChar(drainRate, estimatedWPM, targetTypingRatio = 0.35, fillDivisor = 2.5) {
    if (estimatedWPM <= 0) return 0;
    const charsPerSecond = estimatedWPM * 5 / 60;
    const fillRate = drainRate / targetTypingRatio;
    return fillRate / charsPerSecond / fillDivisor;
  }
  function getPowerBarColor(power) {
    if (power > 60) return "green";
    if (power > 30) return "yellow";
    if (power > 10) return "red";
    return "critical";
  }
  function isOnWatchPage() {
    const pathname = window.location.pathname;
    return pathname.includes("/watch") || pathname.includes("/shorts/");
  }
  function isVideoPlaying() {
    const video = state.videoElement;
    return video !== null && !video.paused && !video.ended;
  }
  var FINGER_COLORS = {
    "left-pinky": "#ec4899",
    "left-ring": "#f97316",
    "left-middle": "#eab308",
    "left-index": "#22c55e",
    "right-index": "#22c55e",
    "right-middle": "#eab308",
    "right-ring": "#f97316",
    "right-pinky": "#ec4899",
    thumb: "#8b5cf6"
  };
  var KEYBOARD_LAYOUT = [
    [
      { key: "q", display: "Q", finger: "left-pinky", isHomeKey: false },
      { key: "w", display: "W", finger: "left-ring", isHomeKey: false },
      { key: "e", display: "E", finger: "left-middle", isHomeKey: false },
      { key: "r", display: "R", finger: "left-index", isHomeKey: false },
      { key: "t", display: "T", finger: "left-index", isHomeKey: false },
      { key: "y", display: "Y", finger: "right-index", isHomeKey: false },
      { key: "u", display: "U", finger: "right-index", isHomeKey: false },
      { key: "i", display: "I", finger: "right-middle", isHomeKey: false },
      { key: "o", display: "O", finger: "right-ring", isHomeKey: false },
      { key: "p", display: "P", finger: "right-pinky", isHomeKey: false }
    ],
    [
      { key: "a", display: "A", finger: "left-pinky", isHomeKey: false },
      { key: "s", display: "S", finger: "left-ring", isHomeKey: false },
      { key: "d", display: "D", finger: "left-middle", isHomeKey: false },
      { key: "f", display: "F", finger: "left-index", isHomeKey: true },
      { key: "g", display: "G", finger: "left-index", isHomeKey: false },
      { key: "h", display: "H", finger: "right-index", isHomeKey: false },
      { key: "j", display: "J", finger: "right-index", isHomeKey: true },
      { key: "k", display: "K", finger: "right-middle", isHomeKey: false },
      { key: "l", display: "L", finger: "right-ring", isHomeKey: false },
      { key: ";", display: ";", finger: "right-pinky", isHomeKey: false }
    ],
    [
      { key: "z", display: "Z", finger: "left-pinky", isHomeKey: false },
      { key: "x", display: "X", finger: "left-ring", isHomeKey: false },
      { key: "c", display: "C", finger: "left-middle", isHomeKey: false },
      { key: "v", display: "V", finger: "left-index", isHomeKey: false },
      { key: "b", display: "B", finger: "left-index", isHomeKey: false },
      { key: "n", display: "N", finger: "right-index", isHomeKey: false },
      { key: "m", display: "M", finger: "right-index", isHomeKey: false },
      { key: ",", display: ",", finger: "right-middle", isHomeKey: false },
      { key: ".", display: ".", finger: "right-ring", isHomeKey: false },
      { key: "/", display: "/", finger: "right-pinky", isHomeKey: false }
    ],
    [{ key: " ", display: "Space", finger: "thumb", isHomeKey: false, width: 10 }]
  ];
  var DEV_MODE = true;
  var powerBarState = {
    currentPower: DEV_MODE ? 5 : 100,
    // Start very low in dev mode for quick testing
    maxPower: 100,
    drainRate: 0,
    baseFillPerChar: 0,
    estimatedWPM: 20,
    // Default for beginners
    targetTypingRatio: 0.35,
    isDraining: false,
    isEmpty: false,
    isFull: DEV_MODE ? false : true
  };
  var letterProgressState = null;
  var recentWords = /* @__PURE__ */ new Set();
  var MAX_RECENT_WORDS = 10;
  var captionWords = [];
  var lastVideoId = null;
  var CAPTION_WORD_RATIO = 0.7;
  var currentBrainrotFactor = 1;
  var currentVideoTitle = "";
  var currentChannelName = "";
  var recentSentences = /* @__PURE__ */ new Set();
  var MAX_RECENT_SENTENCES = 20;
  function createEmptySession() {
    return {
      targetText: "",
      characters: [],
      currentIndex: 0,
      streak: 0,
      maxStreak: 0,
      correctCount: 0,
      errorCount: 0,
      correctedCount: 0,
      startTime: null,
      isComplete: false,
      hasIncorrectChar: false,
      letterTimings: /* @__PURE__ */ new Map()
    };
  }
  var state = {
    isActive: false,
    session: createEmptySession(),
    videoElement: null,
    keyboardVisible: true,
    lastCelebratedStreak: 0
  };
  var overlay = null;
  var miniOverlay = null;
  var powerBarElement = null;
  var drainInterval = null;
  var statsUpdateInterval = null;
  var wordsPerChallenge = 12;
  var lastKeypressTime = null;
  var miniModeEnabled = true;
  var sessionDrillsCompleted = 0;
  var currentUserId = null;
  var DRAIN_TICK_MS = 100;
  function initializePowerBar(estimatedWPM = 20, targetTypingRatio = 0.35) {
    powerBarState.estimatedWPM = estimatedWPM;
    powerBarState.targetTypingRatio = targetTypingRatio;
    powerBarState.drainRate = calculateDrainRate(estimatedWPM, targetTypingRatio);
    powerBarState.baseFillPerChar = calculateBaseFillPerChar(
      powerBarState.drainRate,
      estimatedWPM,
      targetTypingRatio
    );
    if (!DEV_MODE) {
      powerBarState.currentPower = powerBarState.maxPower;
      powerBarState.isEmpty = false;
      powerBarState.isFull = true;
    }
    console.log("[TypeTube] Power bar initialized:", {
      devMode: DEV_MODE,
      currentPower: powerBarState.currentPower,
      estimatedWPM,
      drainRate: powerBarState.drainRate,
      baseFillPerChar: powerBarState.baseFillPerChar,
      timeToEmpty: powerBarState.currentPower / powerBarState.drainRate + " seconds"
    });
  }
  function startDrain() {
    if (powerBarState.isDraining) return;
    if (!isOnWatchPage()) {
      console.log("[TypeTube] Not on watch page, skipping drain");
      return;
    }
    if (!isVideoPlaying()) {
      console.log("[TypeTube] No video playing, skipping drain");
      return;
    }
    powerBarState.isDraining = true;
    if (drainInterval) {
      clearInterval(drainInterval);
    }
    drainInterval = setInterval(() => {
      if (!powerBarState.isDraining) return;
      if (!isOnWatchPage() || !isVideoPlaying()) {
        stopDrain();
        return;
      }
      const effectiveDrainRate = powerBarState.drainRate * currentBrainrotFactor;
      const drainAmount = effectiveDrainRate * DRAIN_TICK_MS / 1e3;
      powerBarState.currentPower = Math.max(0, powerBarState.currentPower - drainAmount);
      powerBarState.isEmpty = powerBarState.currentPower <= 0;
      powerBarState.isFull = false;
      updatePowerBarDisplay();
      if (powerBarState.isEmpty && state.videoElement && !state.isActive) {
        stopDrain();
        startTypingChallenge();
      }
    }, DRAIN_TICK_MS);
  }
  function stopDrain() {
    if (drainInterval) {
      clearInterval(drainInterval);
      drainInterval = null;
    }
    powerBarState.isDraining = false;
  }
  function addPowerForCorrectChar() {
    const session = state.session;
    const accuracy = getSessionAccuracy();
    const fillAmount = calculateFillAmount(
      powerBarState.baseFillPerChar,
      accuracy,
      session.streak
    );
    powerBarState.currentPower = Math.min(powerBarState.maxPower, powerBarState.currentPower + fillAmount);
    powerBarState.isEmpty = false;
    powerBarState.isFull = powerBarState.currentPower >= powerBarState.maxPower;
    updatePowerBarDisplay();
    updateOverlayPowerBar();
    const fillElement = document.getElementById("typetube-overlay-power-fill");
    if (fillElement) {
      fillElement.classList.add("filling");
      setTimeout(() => fillElement.classList.remove("filling"), 300);
    }
  }
  function applyErrorPenalty() {
    const penalty = calculateErrorPenalty(powerBarState.baseFillPerChar);
    powerBarState.currentPower = Math.max(0, powerBarState.currentPower - penalty);
    powerBarState.isEmpty = powerBarState.currentPower <= 0;
    powerBarState.isFull = false;
    updatePowerBarDisplay();
    updateOverlayPowerBar();
  }
  function getSessionAccuracy() {
    const session = state.session;
    const total = session.correctCount + session.errorCount;
    if (total === 0) return 100;
    return Math.round(session.correctCount / total * 100);
  }
  function createPowerBarElement() {
    if (powerBarElement) {
      return;
    }
    powerBarElement = document.createElement("div");
    powerBarElement.id = "typetube-power-bar";
    const initialPower = Math.round(powerBarState.currentPower / powerBarState.maxPower * 100);
    powerBarElement.innerHTML = `
    <div class="typetube-power-bar-label">TypeTube</div>
    <div class="typetube-power-bar-container">
      <div class="typetube-power-bar-fill" id="typetube-power-fill" style="width: ${initialPower}%"></div>
      <div class="typetube-power-bar-text" id="typetube-power-text">${initialPower}%</div>
    </div>
    <div class="typetube-power-bar-bf" id="typetube-powerbar-bf">BF: 1x</div>
  `;
    document.body.appendChild(powerBarElement);
    injectStyles();
    updatePowerBarDisplay();
    console.log("[TypeTube] Power bar element created");
  }
  function updatePowerBarDisplay() {
    const fillElement = document.getElementById("typetube-power-fill");
    const textElement = document.getElementById("typetube-power-text");
    const containerElement = document.getElementById("typetube-power-bar");
    const percentage = Math.round(powerBarState.currentPower / powerBarState.maxPower * 100);
    if (fillElement) {
      const color = getPowerBarColor(percentage);
      fillElement.style.width = `${percentage}%`;
      fillElement.classList.remove("power-green", "power-yellow", "power-red", "power-critical");
      fillElement.classList.add(`power-${color}`);
    }
    if (textElement) {
      textElement.textContent = `${percentage}%`;
    }
    if (containerElement) {
      containerElement.classList.remove("power-full", "power-low", "power-critical-state");
      if (percentage >= 95) {
        containerElement.classList.add("power-full");
      } else if (percentage <= 10) {
        containerElement.classList.add("power-critical-state");
      } else if (percentage <= 30) {
        containerElement.classList.add("power-low");
      }
    }
  }
  function resetSession(text) {
    const characters = text.split("").map((char) => ({
      char,
      status: "pending"
    }));
    state.session = {
      targetText: text,
      characters,
      currentIndex: 0,
      streak: 0,
      maxStreak: 0,
      correctCount: 0,
      errorCount: 0,
      correctedCount: 0,
      startTime: null,
      isComplete: false,
      hasIncorrectChar: false,
      letterTimings: /* @__PURE__ */ new Map()
    };
    state.lastCelebratedStreak = 0;
  }
  function handleKeyPress(key) {
    const session = state.session;
    const now = Date.now();
    if (session.startTime === null && session.targetText.length > 0) {
      session.startTime = now;
      lastKeypressTime = now;
    }
    if (session.isComplete || session.targetText.length === 0) {
      return false;
    }
    if (session.hasIncorrectChar) {
      return false;
    }
    if (session.currentIndex >= session.characters.length) {
      return false;
    }
    const currentChar = session.characters[session.currentIndex];
    const expectedChar = currentChar.char;
    const timeSinceLastKey = lastKeypressTime ? now - lastKeypressTime : 0;
    lastKeypressTime = now;
    if (key === expectedChar) {
      const isCorrection = currentChar.status === "incorrect";
      if (isCorrection) {
        currentChar.status = "corrected";
        session.correctedCount++;
      } else {
        currentChar.status = "correct";
      }
      currentChar.typedChar = key;
      session.correctCount++;
      session.currentIndex++;
      session.streak++;
      if (!isCorrection && timeSinceLastKey > 0 && timeSinceLastKey < 5e3) {
        session.letterTimings.set(expectedChar.toLowerCase(), timeSinceLastKey);
      }
      if (letterProgressState && !isCorrection) {
        if (/^[a-zA-Z]$/.test(expectedChar)) {
          recordKeystroke(letterProgressState, expectedChar.toLowerCase(), true, timeSinceLastKey);
        }
        if (isCapitalLetter(expectedChar)) {
          recordCapitalKeystroke(letterProgressState, true);
        }
        if (isPunctuation(expectedChar)) {
          recordPunctuationKeystroke(letterProgressState, true);
        }
      }
      if (session.streak > session.maxStreak) {
        session.maxStreak = session.streak;
      }
      addPowerForCorrectChar();
      if (session.currentIndex >= session.characters.length) {
        session.isComplete = true;
      }
      return true;
    } else {
      currentChar.status = "incorrect";
      currentChar.typedChar = key;
      session.errorCount++;
      session.streak = 0;
      session.hasIncorrectChar = true;
      if (letterProgressState) {
        if (/^[a-zA-Z]$/.test(expectedChar)) {
          recordKeystroke(letterProgressState, expectedChar.toLowerCase(), false, 0);
        }
        if (isCapitalLetter(expectedChar)) {
          recordCapitalKeystroke(letterProgressState, false);
        }
        if (isPunctuation(expectedChar)) {
          recordPunctuationKeystroke(letterProgressState, false);
        }
      }
      applyErrorPenalty();
      return true;
    }
  }
  function handleBackspace() {
    const session = state.session;
    if (session.hasIncorrectChar && session.currentIndex < session.characters.length) {
      const currentChar = session.characters[session.currentIndex];
      currentChar.status = "pending";
      currentChar.typedChar = void 0;
      session.hasIncorrectChar = false;
      return true;
    }
    return false;
  }
  function getWPM() {
    const session = state.session;
    if (!session.startTime || session.correctCount === 0) return 0;
    const elapsedMs = Date.now() - session.startTime;
    const elapsedMinutes = elapsedMs / 6e4;
    if (elapsedMinutes <= 0) return 0;
    const words = session.correctCount / 5;
    return Math.round(words / elapsedMinutes);
  }
  function checkAndRedirectShorts() {
    const path = window.location.pathname;
    if (path.startsWith("/shorts/") || path === "/shorts") {
      console.log("[TypeTube] Shorts URL detected, redirecting to homepage");
      window.location.href = "/";
      return true;
    }
    return false;
  }
  function observeShortsNavigation() {
    if (checkAndRedirectShorts()) return;
    const originalPushState = history.pushState;
    const originalReplaceState = history.replaceState;
    history.pushState = function(...args) {
      originalPushState.apply(this, args);
      checkAndRedirectShorts();
    };
    history.replaceState = function(...args) {
      originalReplaceState.apply(this, args);
      checkAndRedirectShorts();
    };
    window.addEventListener("popstate", () => {
      checkAndRedirectShorts();
    });
    window.addEventListener("yt-navigate-finish", () => {
      checkAndRedirectShorts();
    });
  }
  function injectShortsBlockingStyles() {
    const styleId = "typetube-shorts-blocker";
    if (document.getElementById(styleId)) return;
    const style = document.createElement("style");
    style.id = styleId;
    style.textContent = `
    /* Hide Shorts shelf on homepage - primary selectors that work */
    ytd-rich-shelf-renderer[is-shorts],
    ytd-rich-section-renderer:has(ytd-rich-shelf-renderer[is-shorts]),

    /* Hide individual Shorts thumbnails anywhere on the page */
    ytd-rich-item-renderer:has(a[href*="/shorts/"]),
    ytd-video-renderer:has(a[href*="/shorts/"]),
    ytd-grid-video-renderer:has(a[href*="/shorts/"]),
    ytd-compact-video-renderer:has(a[href*="/shorts/"]),

    /* Hide Shorts in sidebar/navigation */
    ytd-guide-entry-renderer:has(a[title="Shorts"]),
    ytd-mini-guide-entry-renderer:has(a[title="Shorts"]),

    /* Hide Shorts tab on channel pages */
    yt-tab-shape[tab-title="Shorts"],
    tp-yt-paper-tab:has(a[href*="/shorts"]),
    yt-chip-cloud-chip-renderer:has([title="Shorts"]),

    /* Hide Shorts in search results */
    ytd-reel-item-renderer,
    ytd-reel-shelf-renderer,

    /* Hide any element with is-shorts attribute */
    [is-shorts]:not(style),

    /* Hide Shorts overlay badges */
    ytd-thumbnail-overlay-time-status-renderer[overlay-style="SHORTS"],
    ytd-rich-item-renderer:has([overlay-style="SHORTS"]),
    ytd-video-renderer:has([overlay-style="SHORTS"]),

    /* Hide Shorts remixing elements */
    ytd-reel-video-renderer {
      display: none !important;
    }
  `;
    (document.head || document.documentElement).appendChild(style);
    console.log("[TypeTube] Shorts blocking styles injected");
  }
  injectShortsBlockingStyles();
  observeShortsNavigation();
  function init() {
    console.log("[TypeTube] Content script loaded (v3 with KeyBr word generation)");
    loadCurrentUser();
    loadSettings();
    loadLetterProgressAsync();
    initializePowerBar();
    observeVideoElement();
    injectStyles();
    initThumbnailObserver();
    chrome.runtime.onMessage.addListener((message) => {
      if (message.type === "USER_CHANGED") {
        currentUserId = message.userId;
        console.log("[TypeTube] User changed to:", currentUserId);
      }
    });
  }
  async function loadCurrentUser() {
    try {
      const result = await chrome.runtime.sendMessage({ type: "GET_CURRENT_USER" });
      if (result?.user) {
        currentUserId = result.user.id;
        console.log("[TypeTube] Current user loaded:", currentUserId);
      }
    } catch (e) {
      console.error("[TypeTube] Error loading current user:", e);
    }
  }
  async function loadLetterProgressAsync() {
    try {
      letterProgressState = await loadLetterProgress();
      const summary = getProgressSummary(letterProgressState);
      console.log(
        `[TypeTube] Letter progress loaded: ${summary.unlockedCount}/${summary.totalLetters} letters, emphasis: ${summary.emphasisLetter?.toUpperCase() || "none"}`
      );
    } catch (e) {
      console.error("[TypeTube] Error loading letter progress:", e);
    }
  }
  async function loadSettings() {
    try {
      const result = await chrome.storage.sync.get([
        "wordsPerChallenge",
        "estimatedWPM",
        "targetTypingRatio"
      ]);
      if (result.wordsPerChallenge) wordsPerChallenge = result.wordsPerChallenge;
      if (result.estimatedWPM) powerBarState.estimatedWPM = result.estimatedWPM;
      if (result.targetTypingRatio) powerBarState.targetTypingRatio = result.targetTypingRatio;
      initializePowerBar(powerBarState.estimatedWPM, powerBarState.targetTypingRatio);
    } catch (e) {
      console.error("[TypeTube] Error loading settings:", e);
    }
  }
  async function loadCaptionsForCurrentVideo() {
    const videoId = getVideoIdFromUrl();
    if (!videoId) {
      console.log("[TypeTube] No video ID found in URL");
      captionWords = [];
      lastVideoId = null;
      return;
    }
    if (videoId === lastVideoId && captionWords.length > 0) {
      console.log(`[TypeTube] Captions already loaded for video ${videoId}`);
      return;
    }
    console.log(`[TypeTube] Loading captions for video: ${videoId}`);
    lastVideoId = videoId;
    try {
      await new Promise((resolve) => setTimeout(resolve, 1e3));
      captionWords = await extractCaptions(videoId);
      console.log(`[TypeTube] Loaded ${captionWords.length} caption words for video ${videoId}`);
      if (captionWords.length === 0) {
        console.log("[TypeTube] No captions available, will use word list only");
      }
    } catch (e) {
      console.error("[TypeTube] Error loading captions:", e);
      captionWords = [];
    }
  }
  async function loadBrainrotFactorForCurrentVideo() {
    const videoId = getVideoIdFromUrl();
    if (!videoId) {
      currentBrainrotFactor = 1;
      currentVideoTitle = "";
      currentChannelName = "";
      updateBrainrotDisplay();
      return;
    }
    await new Promise((resolve) => setTimeout(resolve, 500));
    let metadata = extractCurrentVideoMetadata(videoId);
    if (metadata.title === "Unknown Video" || metadata.channelName === "Unknown Channel") {
      await new Promise((resolve) => setTimeout(resolve, 1e3));
      metadata = extractCurrentVideoMetadata(videoId);
    }
    currentVideoTitle = metadata.title;
    currentChannelName = metadata.channelName;
    console.log(`[TypeTube] Classifying: "${metadata.title}" by ${metadata.channelName}`);
    try {
      const result = await chrome.runtime.sendMessage({
        type: "CLASSIFY_VIDEOS",
        videos: [metadata]
      });
      if (result?.success && result.results?.length > 0) {
        currentBrainrotFactor = result.results[0].factor;
        console.log(`[TypeTube] Brainrot factor: ${currentBrainrotFactor}x - ${result.results[0].reasoning || "No reasoning"}`);
      } else {
        console.log("[TypeTube] Classification failed, using default BF 1x");
        currentBrainrotFactor = 1;
      }
    } catch (e) {
      console.error("[TypeTube] Error getting brainrot factor:", e);
      currentBrainrotFactor = 1;
    }
    updateBrainrotDisplay();
  }
  function extractCurrentVideoMetadata(videoId) {
    const titleSelectors = [
      "h1.ytd-watch-metadata yt-formatted-string",
      "#title h1 yt-formatted-string",
      "h1.ytd-video-primary-info-renderer yt-formatted-string",
      "ytd-watch-metadata h1 yt-formatted-string",
      "#above-the-fold #title yt-formatted-string",
      "h1.title"
    ];
    let title = "Unknown Video";
    for (const selector of titleSelectors) {
      const el = document.querySelector(selector);
      if (el?.textContent?.trim()) {
        title = el.textContent.trim();
        break;
      }
    }
    if (title === "Unknown Video") {
      const docTitle = document.title;
      if (docTitle && docTitle !== "YouTube" && docTitle.endsWith(" - YouTube")) {
        title = docTitle.slice(0, -10).trim();
      }
    }
    const channelSelectors = [
      "ytd-watch-metadata ytd-channel-name a",
      "#channel-name a",
      "#owner ytd-channel-name a",
      "ytd-channel-name yt-formatted-string a",
      "#upload-info ytd-channel-name a",
      "ytd-video-owner-renderer #channel-name a"
    ];
    let channelName = "Unknown Channel";
    for (const selector of channelSelectors) {
      const el = document.querySelector(selector);
      if (el?.textContent?.trim()) {
        channelName = el.textContent.trim();
        break;
      }
    }
    const descSelectors = [
      "#description-inline-expander yt-attributed-string",
      "#description yt-formatted-string",
      "ytd-text-inline-expander #attributed-snippet-text",
      "#attributed-snippet-text"
    ];
    let description;
    for (const selector of descSelectors) {
      const el = document.querySelector(selector);
      if (el?.textContent?.trim()) {
        description = el.textContent.trim().slice(0, 500);
        break;
      }
    }
    const durationElement = document.querySelector(".ytp-time-duration");
    const duration = durationElement?.textContent?.trim();
    const viewElement = document.querySelector("#info-strings yt-formatted-string, ytd-video-view-count-renderer span");
    const viewCount = viewElement?.textContent?.trim();
    const isShort = window.location.pathname.includes("/shorts/");
    const adjustedDuration = isShort ? "SHORTS" : duration;
    return {
      videoId,
      title,
      channelName,
      description,
      duration: adjustedDuration,
      viewCount
    };
  }
  function updateBrainrotDisplay() {
    const bfDisplay = document.getElementById("typetube-bf-display");
    if (bfDisplay) {
      bfDisplay.textContent = `${currentBrainrotFactor}x`;
      bfDisplay.style.color = getBrainrotColor(currentBrainrotFactor);
    }
    const powerBarBf = document.getElementById("typetube-powerbar-bf");
    if (powerBarBf) {
      powerBarBf.textContent = `BF: ${currentBrainrotFactor}x`;
      powerBarBf.style.color = getBrainrotColor(currentBrainrotFactor);
    }
  }
  function getBrainrotColor(factor) {
    if (factor >= 6) return "#ef4444";
    if (factor >= 4) return "#f97316";
    if (factor >= 2) return "#eab308";
    if (factor >= 1.25) return "#a3a3a3";
    if (factor >= 1) return "#22c55e";
    if (factor >= 0.5) return "#3b82f6";
    return "#8b5cf6";
  }
  function getBrainrotLabel(factor) {
    if (factor >= 10) return "EXT";
    if (factor >= 6) return "VHI";
    if (factor >= 4) return "HI";
    if (factor >= 2.5) return "M+";
    if (factor >= 2) return "M";
    if (factor >= 1.5) return "M-";
    if (factor >= 1.25) return "OK+";
    if (factor >= 1) return "OK";
    if (factor >= 0.75) return "GD";
    if (factor >= 0.5) return "GR";
    return "EDU";
  }
  var processedThumbnails = /* @__PURE__ */ new Set();
  var thumbnailClassificationQueue = [];
  var classificationDebounceTimer = null;
  function extractVideoIdFromHref(href) {
    const watchMatch = href.match(/[?&]v=([^&]+)/);
    if (watchMatch) return watchMatch[1];
    const shortsMatch = href.match(/\/shorts\/([^?&/]+)/);
    if (shortsMatch) return shortsMatch[1];
    return null;
  }
  function processThumbnails() {
    const thumbnailSelectors = [
      // New YouTube structure (2024+)
      "yt-lockup-view-model",
      // New lockup component used on homepage
      // Legacy structures (still used in some places)
      "ytd-thumbnail",
      // Standard video thumbnails
      "ytd-rich-item-renderer #thumbnail",
      // Homepage grid (legacy)
      "ytd-compact-video-renderer #thumbnail",
      // Sidebar recommendations
      "ytd-video-renderer #thumbnail"
      // Search results
    ];
    const thumbnails = document.querySelectorAll(thumbnailSelectors.join(", "));
    if (thumbnails.length > 0) {
      console.log(`[TypeTube] Found ${thumbnails.length} thumbnails to process`);
    }
    for (const thumbnail of thumbnails) {
      processSingleThumbnail(thumbnail);
    }
  }
  function processSingleThumbnail(thumbnail) {
    let link = null;
    if (thumbnail.tagName === "A" && thumbnail.getAttribute("href")) {
      link = thumbnail;
    }
    if (!link) {
      link = thumbnail.querySelector("a.yt-lockup-view-model__content-image[href]");
    }
    if (!link) {
      link = thumbnail.querySelector("a#thumbnail[href]");
    }
    if (!link) {
      link = thumbnail.querySelector('a[href*="/watch"], a[href*="/shorts"]');
    }
    if (!link) {
      link = thumbnail.querySelector("a[href]");
    }
    if (!link) {
      const parent = thumbnail.parentElement;
      if (parent?.tagName === "A" && parent.getAttribute("href")) {
        link = parent;
      }
    }
    if (!link) {
      const renderer = thumbnail.closest("ytd-rich-item-renderer, ytd-compact-video-renderer, ytd-video-renderer, ytd-grid-video-renderer, ytd-rich-grid-media");
      if (renderer) {
        link = renderer.querySelector('a.yt-lockup-view-model__content-image[href], a#thumbnail[href], a[href*="/watch"], a[href*="/shorts"]');
      }
    }
    if (!link) {
      const innerPreview = thumbnail.innerHTML.substring(0, 200);
      console.log("[TypeTube] No link found in:", thumbnail.tagName, "- Inner preview:", innerPreview);
      return;
    }
    const href = link.getAttribute("href") || "";
    const videoId = extractVideoIdFromHref(href);
    if (!videoId) {
      console.log("[TypeTube] Could not extract video ID from:", href);
      return;
    }
    if (processedThumbnails.has(videoId)) {
      return;
    }
    processedThumbnails.add(videoId);
    let thumbnailContainer = null;
    thumbnailContainer = thumbnail.querySelector("yt-thumbnail-view-model, a.yt-lockup-view-model__content-image");
    if (!thumbnailContainer) {
      thumbnailContainer = thumbnail.querySelector("#thumbnail");
    }
    if (!thumbnailContainer) {
      thumbnailContainer = thumbnail;
    }
    const container = thumbnailContainer;
    const currentPosition = window.getComputedStyle(container).position;
    if (currentPosition === "static") {
      container.style.position = "relative";
    }
    const metadata = extractThumbnailMetadata(thumbnail, videoId);
    chrome.runtime.sendMessage({ type: "GET_CACHED_BRAINROT", videoId }).then((result) => {
      if (result?.factor !== null && result?.factor !== void 0) {
        addBadgeToThumbnail(container, result.factor);
      } else {
        queueThumbnailForClassification(container, metadata);
      }
    }).catch(() => {
      queueThumbnailForClassification(container, metadata);
    });
  }
  function extractThumbnailMetadata(thumbnail, videoId) {
    const renderer = thumbnail.closest("ytd-rich-item-renderer, ytd-compact-video-renderer, ytd-video-renderer, ytd-grid-video-renderer, yt-lockup-view-model");
    let title = "Unknown Video";
    const titleElement = renderer?.querySelector("h3, #video-title, #video-title-link, a#video-title");
    if (titleElement) {
      title = titleElement.getAttribute("title") || titleElement.textContent?.trim() || title;
    }
    let channelName = "Unknown Channel";
    const channelElement = renderer?.querySelector('a[href*="/@"], #channel-name a, ytd-channel-name a, .ytd-channel-name a');
    if (channelElement) {
      channelName = channelElement.textContent?.trim() || channelName;
    }
    return { videoId, title, channelName };
  }
  function queueThumbnailForClassification(thumbnailElement, metadata) {
    console.log(`[TypeTube] Queuing thumbnail: "${metadata.title}" by ${metadata.channelName}`);
    thumbnailClassificationQueue.push({
      ...metadata,
      thumbnailElement
    });
    if (classificationDebounceTimer) {
      clearTimeout(classificationDebounceTimer);
    }
    classificationDebounceTimer = setTimeout(() => {
      processClassificationQueue();
    }, 500);
  }
  async function processClassificationQueue() {
    if (thumbnailClassificationQueue.length === 0) return;
    const batch = [...thumbnailClassificationQueue];
    thumbnailClassificationQueue = [];
    console.log(`[TypeTube] Processing ${batch.length} thumbnails for classification`);
    try {
      const videos = batch.map((item) => ({
        videoId: item.videoId,
        title: item.title,
        channelName: item.channelName
      }));
      const result = await chrome.runtime.sendMessage({
        type: "CLASSIFY_VIDEOS",
        videos
      });
      if (result?.success && result.results) {
        console.log(`[TypeTube] Got ${result.results.length} classification results`);
        for (const item of batch) {
          const classification = result.results.find((r) => r.videoId === item.videoId);
          if (classification) {
            let container = item.thumbnailElement;
            if (!document.body.contains(container)) {
              console.log(`[TypeTube] Element detached for ${item.videoId}, re-finding...`);
              const newContainer = findThumbnailByVideoId(item.videoId);
              if (newContainer) {
                container = newContainer;
              } else {
                console.log(`[TypeTube] Could not re-find thumbnail for ${item.videoId}`);
                continue;
              }
            }
            addBadgeToThumbnail(container, classification.factor);
          }
        }
      } else {
        console.log("[TypeTube] Classification failed:", result);
      }
    } catch (error) {
      console.error("[TypeTube] Thumbnail classification error:", error);
    }
  }
  function findThumbnailByVideoId(videoId) {
    const links = document.querySelectorAll(`a[href*="${videoId}"]`);
    for (const link of links) {
      const lockup = link.closest("yt-lockup-view-model");
      if (lockup) {
        const container = lockup.querySelector("yt-thumbnail-view-model, a.yt-lockup-view-model__content-image");
        if (container) return container;
      }
      const renderer = link.closest("ytd-rich-item-renderer, ytd-compact-video-renderer, ytd-video-renderer");
      if (renderer) {
        const container = renderer.querySelector("#thumbnail, ytd-thumbnail");
        if (container) return container;
      }
    }
    return null;
  }
  function addBadgeToThumbnail(container, factor) {
    if (container.querySelector(".typetube-bf-badge")) {
      return;
    }
    if (!document.body.contains(container)) {
      console.log("[TypeTube] Badge container no longer in DOM, skipping");
      return;
    }
    const currentPosition = window.getComputedStyle(container).position;
    if (currentPosition === "static") {
      container.style.position = "relative";
    }
    const badge = document.createElement("div");
    badge.className = "typetube-bf-badge";
    badge.textContent = `${factor}x`;
    badge.title = `Brainrot Factor: ${factor}x - ${getBrainrotLabel(factor)}`;
    badge.style.cssText = `
    position: absolute;
    top: 4px;
    right: 4px;
    padding: 2px 6px;
    font-size: 12px;
    font-weight: 700;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    color: white;
    background: ${getBrainrotColor(factor)};
    border-radius: 4px;
    z-index: 9999;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.5);
    text-shadow: 0 1px 1px rgba(0, 0, 0, 0.3);
    pointer-events: auto;
  `;
    container.appendChild(badge);
    console.log(`[TypeTube] Added ${factor}x badge to thumbnail`);
  }
  function initThumbnailObserver() {
    processThumbnails();
    const thumbnailObserver = new MutationObserver((mutations) => {
      let hasNewThumbnails = false;
      for (const mutation of mutations) {
        if (mutation.addedNodes.length > 0) {
          for (const node of mutation.addedNodes) {
            if (node instanceof HTMLElement) {
              if (node.matches?.("ytd-thumbnail, ytd-rich-item-renderer, ytd-compact-video-renderer, ytd-video-renderer")) {
                hasNewThumbnails = true;
                break;
              }
              if (node.querySelector?.("ytd-thumbnail, ytd-rich-item-renderer")) {
                hasNewThumbnails = true;
                break;
              }
            }
          }
        }
        if (hasNewThumbnails) break;
      }
      if (hasNewThumbnails) {
        setTimeout(processThumbnails, 100);
      }
    });
    thumbnailObserver.observe(document.body, {
      childList: true,
      subtree: true
    });
  }
  var currentUrl = window.location.href;
  function observeVideoElement() {
    const observer = new MutationObserver(() => {
      if (state.isActive) return;
      if (window.location.href !== currentUrl) {
        const newVideoId = getVideoIdFromUrl();
        const oldVideoId = lastVideoId;
        currentUrl = window.location.href;
        if (newVideoId && newVideoId !== oldVideoId) {
          loadCaptionsForCurrentVideo();
          loadBrainrotFactorForCurrentVideo();
        }
      }
      const video2 = document.querySelector("video.html5-main-video");
      if (video2 && video2 !== state.videoElement) {
        state.videoElement = video2;
        listenersSetUp = false;
        setupVideoListeners(video2);
        createPowerBarElement();
      }
    });
    observer.observe(document.body, {
      childList: true,
      subtree: true
    });
    window.addEventListener("popstate", () => {
      const newVideoId = getVideoIdFromUrl();
      if (newVideoId && newVideoId !== lastVideoId) {
        console.log(`[TypeTube] Navigation detected, new video: ${newVideoId}`);
        loadCaptionsForCurrentVideo();
        loadBrainrotFactorForCurrentVideo();
      }
    });
    const video = document.querySelector("video.html5-main-video");
    if (video) {
      state.videoElement = video;
      setupVideoListeners(video);
      createPowerBarElement();
    }
  }
  var listenersSetUp = false;
  function setupVideoListeners(video) {
    if (listenersSetUp) {
      return;
    }
    listenersSetUp = true;
    loadCaptionsForCurrentVideo();
    loadBrainrotFactorForCurrentVideo();
    video.addEventListener("play", () => {
      if (state.isActive) {
        video.pause();
        return;
      }
      if (!powerBarState.isEmpty) {
        startDrain();
        if (miniModeEnabled && !miniOverlay) {
          showMiniOverlay();
        }
      }
    });
    video.addEventListener("pause", () => {
      if (!state.isActive) {
        stopDrain();
        if (!powerBarState.isEmpty) {
          hideMiniOverlay();
        }
      }
    });
    if (!video.paused && !state.isActive && !powerBarState.isEmpty) {
      startDrain();
    }
  }
  function generateChallenge() {
    if (!letterProgressState) {
      return "the cat ran fast";
    }
    const stage = getCurrentStage(letterProgressState);
    console.log(`[TypeTube] Current progression stage: ${stage}`);
    if (stage !== "letters") {
      return generateSentenceChallenge(stage);
    }
    const words = selectWords(wordsPerChallenge);
    return words.join(" ");
  }
  function generateSentenceChallenge(stage) {
    const sentence = selectRandomSentence(stage, recentSentences);
    if (!sentence) {
      console.log("[TypeTube] No sentences available, falling back to word mode");
      const words = selectWords(wordsPerChallenge);
      return words.join(" ");
    }
    recentSentences.add(sentence.text);
    if (recentSentences.size > MAX_RECENT_SENTENCES) {
      const toRemove = Array.from(recentSentences).slice(0, recentSentences.size - MAX_RECENT_SENTENCES);
      toRemove.forEach((s) => recentSentences.delete(s));
    }
    console.log(`[TypeTube] Generated ${stage} sentence challenge: "${sentence.text}" (${sentence.wordCount} words, category: ${sentence.category})`);
    return sentence.text;
  }
  function selectWords(count) {
    if (!letterProgressState) return [];
    const allowedLetters = letterProgressState.unlockedLetters;
    const emphasisLetter = getEmphasisLetter(letterProgressState);
    const selected = [];
    const availableCaptionWords = filterByUnlockedLetters(captionWords, allowedLetters).filter((w) => w.text.length >= 3 && w.text.length <= 7).filter((w) => !recentWords.has(w.text));
    const captionCount = availableCaptionWords.length > 0 ? Math.ceil(count * CAPTION_WORD_RATIO) : 0;
    if (captionCount > 0) {
      let captionPool = [...availableCaptionWords];
      if (emphasisLetter) {
        const emphasisCaptionWords = filterByContainingLetter(captionPool, emphasisLetter);
        const emphasisFromCaptions = Math.min(Math.ceil(captionCount / 2), emphasisCaptionWords.length);
        for (let i = 0; i < emphasisFromCaptions && emphasisCaptionWords.length > 0; i++) {
          const randomIndex = Math.floor(Math.random() * emphasisCaptionWords.length);
          const word = emphasisCaptionWords.splice(randomIndex, 1)[0];
          selected.push(word.text);
          recentWords.add(word.text);
          captionPool = captionPool.filter((w) => w.text !== word.text);
        }
      }
      while (selected.length < captionCount && captionPool.length > 0) {
        const randomIndex = Math.floor(Math.random() * captionPool.length);
        const word = captionPool.splice(randomIndex, 1)[0];
        selected.push(word.text);
        recentWords.add(word.text);
      }
      console.log(`[TypeTube] Selected ${selected.length} words from captions`);
    }
    const remainingCount = count - selected.length;
    if (remainingCount > 0) {
      let candidateWords = getWordsWithLetters(allowedLetters);
      candidateWords = getWordsByLength(candidateWords, 3, 7);
      candidateWords = candidateWords.filter((word) => !recentWords.has(word.text));
      if (candidateWords.length === 0) {
        recentWords.clear();
        candidateWords = getWordsByLength(getWordsWithLetters(allowedLetters), 3, 7);
      }
      if (emphasisLetter) {
        const emphasisWords = getWordsContaining(allowedLetters, emphasisLetter);
        const emphasisFiltered = getWordsByLength(emphasisWords, 3, 7).filter(
          (word) => !recentWords.has(word.text)
        );
        const emphasisCount = Math.ceil(remainingCount * 0.6);
        for (let i = 0; i < emphasisCount && emphasisFiltered.length > 0; i++) {
          const randomIndex = Math.floor(Math.random() * emphasisFiltered.length);
          const word = emphasisFiltered.splice(randomIndex, 1)[0];
          selected.push(word.text);
          recentWords.add(word.text);
        }
      }
      while (selected.length < count && candidateWords.length > 0) {
        const randomIndex = Math.floor(Math.random() * candidateWords.length);
        const word = candidateWords.splice(randomIndex, 1)[0];
        selected.push(word.text);
        recentWords.add(word.text);
      }
    }
    if (recentWords.size > MAX_RECENT_WORDS) {
      const toRemove = Array.from(recentWords).slice(0, recentWords.size - MAX_RECENT_WORDS);
      toRemove.forEach((word) => recentWords.delete(word));
    }
    return shuffleArray(selected);
  }
  function shuffleArray(array) {
    const result = [...array];
    for (let i = result.length - 1; i > 0; i--) {
      const j = Math.floor(Math.random() * (i + 1));
      [result[i], result[j]] = [result[j], result[i]];
    }
    return result;
  }
  function startTypingChallenge() {
    if (state.isActive) {
      console.log("[TypeTube] Already in typing challenge, ignoring");
      return;
    }
    if (!state.videoElement) {
      console.log("[TypeTube] No video element, cannot start challenge");
      return;
    }
    console.log("[TypeTube] Starting typing challenge");
    hideMiniOverlay();
    state.videoElement.pause();
    stopDrain();
    const challengeText = generateChallenge();
    resetSession(challengeText);
    state.isActive = true;
    showOverlay();
  }
  function injectStyles() {
    const styleId = "typetube-styles-v2";
    if (document.getElementById(styleId)) return;
    const style = document.createElement("style");
    style.id = styleId;
    style.textContent = `
    /* Power bar keyframe animations */
    @keyframes power-gradient-shift {
      0% { background-position: 0% 50%; }
      50% { background-position: 100% 50%; }
      100% { background-position: 0% 50%; }
    }

    @keyframes power-glow-pulse {
      0%, 100% {
        box-shadow: 0 4px 24px rgba(0, 0, 0, 0.6),
                    0 0 20px rgba(139, 92, 246, 0.4),
                    0 0 40px rgba(139, 92, 246, 0.2),
                    inset 0 1px 0 rgba(255, 255, 255, 0.1);
      }
      50% {
        box-shadow: 0 4px 24px rgba(0, 0, 0, 0.6),
                    0 0 30px rgba(139, 92, 246, 0.6),
                    0 0 60px rgba(139, 92, 246, 0.3),
                    inset 0 1px 0 rgba(255, 255, 255, 0.15);
      }
    }

    @keyframes power-low-pulse {
      0%, 100% {
        transform: scale(1);
        filter: brightness(1);
      }
      50% {
        transform: scale(1.01);
        filter: brightness(1.1);
      }
    }

    @keyframes power-critical-shake {
      0%, 100% { transform: translateX(0); }
      10%, 30%, 50%, 70%, 90% { transform: translateX(-1px); }
      20%, 40%, 60%, 80% { transform: translateX(1px); }
    }

    @keyframes shimmer {
      0% { left: -100%; }
      100% { left: 200%; }
    }

    @keyframes particle-float {
      0% {
        transform: translateY(0) scale(1);
        opacity: 1;
      }
      100% {
        transform: translateY(-20px) scale(0);
        opacity: 0;
      }
    }

    @keyframes border-glow {
      0%, 100% { border-color: rgba(139, 92, 246, 0.5); }
      50% { border-color: rgba(167, 139, 250, 0.8); }
    }

    @keyframes pulse-critical {
      0%, 100% { opacity: 1; filter: brightness(1); }
      50% { opacity: 0.7; filter: brightness(1.3); }
    }

    /* Power bar styles - visible floating bar (hidden during drill) */
    #typetube-power-bar {
      position: fixed;
      top: 70px;
      right: 20px;
      width: 220px;
      padding: 12px 16px;
      background: linear-gradient(135deg, rgba(20, 20, 40, 0.98) 0%, rgba(30, 20, 50, 0.98) 100%);
      border-radius: 14px;
      z-index: 2147483647;
      border: 2px solid rgba(139, 92, 246, 0.5);
      box-shadow: 0 4px 24px rgba(0, 0, 0, 0.6),
                  0 0 40px rgba(139, 92, 246, 0.15),
                  inset 0 1px 0 rgba(255, 255, 255, 0.1);
      animation: border-glow 3s ease-in-out infinite;
      backdrop-filter: blur(10px);
    }

    #typetube-power-bar.power-full {
      animation: power-glow-pulse 2s ease-in-out infinite, border-glow 3s ease-in-out infinite;
    }

    #typetube-power-bar.power-low {
      animation: power-low-pulse 1s ease-in-out infinite, border-glow 3s ease-in-out infinite;
    }

    #typetube-power-bar.power-critical-state {
      animation: power-critical-shake 0.5s ease-in-out infinite, border-glow 3s ease-in-out infinite;
      border-color: rgba(239, 68, 68, 0.7) !important;
    }

    #typetube-power-bar.hidden {
      display: none;
    }

    .typetube-power-bar-label {
      font-size: 13px;
      font-weight: 700;
      color: #e9d5ff;
      margin-bottom: 8px;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
      text-transform: uppercase;
      letter-spacing: 2px;
      text-shadow: 0 0 10px rgba(139, 92, 246, 0.5);
      display: flex;
      align-items: center;
      gap: 6px;
    }

    .typetube-power-bar-label::before {
      content: '';
      display: inline-block;
      width: 8px;
      height: 8px;
      background: linear-gradient(135deg, #a78bfa, #8b5cf6);
      border-radius: 50%;
      box-shadow: 0 0 8px rgba(139, 92, 246, 0.8);
      animation: power-glow-pulse 2s ease-in-out infinite;
    }

    .typetube-power-bar-container {
      position: relative;
      height: 18px;
      background: linear-gradient(180deg, rgba(0, 0, 0, 0.4) 0%, rgba(30, 30, 50, 0.6) 100%);
      border-radius: 10px;
      overflow: hidden;
      border: 1px solid rgba(255, 255, 255, 0.1);
      box-shadow: inset 0 2px 4px rgba(0, 0, 0, 0.4);
    }

    .typetube-power-bar-fill {
      position: absolute;
      top: 0;
      left: 0;
      height: 100%;
      border-radius: 9px;
      transition: width 0.15s ease-out, background 0.3s ease;
      overflow: hidden;
    }

    /* Shimmer effect overlay */
    .typetube-power-bar-fill::after {
      content: '';
      position: absolute;
      top: 0;
      left: -100%;
      width: 50%;
      height: 100%;
      background: linear-gradient(
        90deg,
        transparent,
        rgba(255, 255, 255, 0.3),
        transparent
      );
      animation: shimmer 2.5s ease-in-out infinite;
    }

    /* Inner glow overlay */
    .typetube-power-bar-fill::before {
      content: '';
      position: absolute;
      top: 0;
      left: 0;
      right: 0;
      height: 50%;
      background: linear-gradient(180deg, rgba(255, 255, 255, 0.25) 0%, transparent 100%);
      border-radius: 9px 9px 0 0;
    }

    .typetube-power-bar-fill.power-green {
      background: linear-gradient(90deg, #22c55e 0%, #4ade80 25%, #86efac 50%, #4ade80 75%, #22c55e 100%);
      background-size: 200% 100%;
      animation: power-gradient-shift 3s ease infinite;
      box-shadow: 0 0 15px rgba(34, 197, 94, 0.5),
                  inset 0 0 10px rgba(255, 255, 255, 0.2);
    }

    .typetube-power-bar-fill.power-yellow {
      background: linear-gradient(90deg, #eab308 0%, #fde047 25%, #fef08a 50%, #fde047 75%, #eab308 100%);
      background-size: 200% 100%;
      animation: power-gradient-shift 2.5s ease infinite;
      box-shadow: 0 0 15px rgba(234, 179, 8, 0.5),
                  inset 0 0 10px rgba(255, 255, 255, 0.2);
    }

    .typetube-power-bar-fill.power-red {
      background: linear-gradient(90deg, #ef4444 0%, #f87171 25%, #fca5a5 50%, #f87171 75%, #ef4444 100%);
      background-size: 200% 100%;
      animation: power-gradient-shift 2s ease infinite;
      box-shadow: 0 0 15px rgba(239, 68, 68, 0.5),
                  inset 0 0 10px rgba(255, 255, 255, 0.2);
    }

    .typetube-power-bar-fill.power-critical {
      background: linear-gradient(90deg, #dc2626 0%, #ef4444 25%, #f87171 50%, #ef4444 75%, #dc2626 100%);
      background-size: 200% 100%;
      animation: power-gradient-shift 1s ease infinite, pulse-critical 0.4s ease-in-out infinite;
      box-shadow: 0 0 20px rgba(220, 38, 38, 0.7),
                  inset 0 0 10px rgba(255, 255, 255, 0.2);
    }

    .typetube-power-bar-text {
      position: absolute;
      top: 50%;
      left: 50%;
      transform: translate(-50%, -50%);
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
      font-size: 11px;
      font-weight: 800;
      color: white;
      text-shadow: 0 1px 3px rgba(0, 0, 0, 0.9),
                   0 0 8px rgba(0, 0, 0, 0.5);
      z-index: 2;
      letter-spacing: 0.5px;
    }

    .typetube-power-bar-bf {
      margin-top: 10px;
      font-size: 13px;
      font-weight: 700;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
      text-align: center;
      color: #22c55e;
      transition: color 0.3s ease, text-shadow 0.3s ease;
      text-shadow: 0 0 8px currentColor;
      letter-spacing: 1px;
    }

    /* Particle container for special effects */
    .typetube-power-particles {
      position: absolute;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      pointer-events: none;
      overflow: hidden;
      border-radius: 9px;
    }

    .typetube-power-particle {
      position: absolute;
      width: 4px;
      height: 4px;
      background: white;
      border-radius: 50%;
      animation: particle-float 1s ease-out forwards;
      box-shadow: 0 0 6px currentColor;
    }

    /* Overlay styles - full screen drill mode */
    #typetube-overlay {
      position: fixed;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      background: rgba(0, 0, 0, 0.92);
      display: flex;
      z-index: 999999;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    }

    /* Main content area (left side) */
    .typetube-main {
      flex: 1;
      display: flex;
      flex-direction: column;
      justify-content: center;
      align-items: center;
      padding: 2rem;
    }

    /* Stats sidebar (right side) */
    .typetube-sidebar {
      width: 180px;
      background: rgba(30, 27, 75, 0.95);
      border-left: 1px solid rgba(139, 92, 246, 0.3);
      padding: 1.5rem 1rem;
      display: flex;
      flex-direction: column;
      gap: 1.5rem;
    }

    .typetube-sidebar-section {
      display: flex;
      flex-direction: column;
      gap: 0.75rem;
    }

    .typetube-sidebar-label {
      font-size: 0.85rem;
      text-transform: uppercase;
      letter-spacing: 0.1em;
      color: rgba(255, 255, 255, 0.4);
    }

    /* Stats in sidebar */
    .typetube-stat-row {
      display: flex;
      align-items: baseline;
      gap: 0.5rem;
    }

    .typetube-stat-value-large {
      font-size: 2rem;
      font-weight: 700;
      font-variant-numeric: tabular-nums;
      line-height: 1;
    }

    .typetube-stat-unit {
      font-size: 0.85rem;
      text-transform: uppercase;
      color: rgba(255, 255, 255, 0.5);
    }

    .typetube-stat-row.wpm .typetube-stat-value-large { color: #60a5fa; }
    .typetube-stat-row.accuracy .typetube-stat-value-large { color: #4ade80; }
    .typetube-stat-row.streak .typetube-stat-value-large { color: #fbbf24; }

    .typetube-streak-badge {
      display: inline-flex;
      align-items: center;
      gap: 0.25rem;
      font-size: 0.85rem;
      color: #fbbf24;
      background: rgba(251, 191, 36, 0.15);
      padding: 0.15rem 0.4rem;
      border-radius: 0.25rem;
      margin-top: 0.25rem;
    }

    /* Power bar in sidebar */
    .typetube-sidebar-power {
      margin-top: auto;
    }

    .typetube-sidebar-power .typetube-power-bar-container {
      height: 24px;
      border-radius: 8px;
    }

    .typetube-sidebar-power .typetube-power-bar-text {
      font-size: 13px;
    }

    /* Letter progress in sidebar - KeyBr order */
    .typetube-letter-grid-keybr {
      display: grid;
      grid-template-columns: repeat(4, 1fr);
      gap: 3px;
    }

    .typetube-letter-box {
      display: flex;
      align-items: center;
      justify-content: center;
      width: 100%;
      aspect-ratio: 1;
      font-size: 0.9rem;
      font-weight: 700;
      font-family: 'Fira Code', monospace;
      border-radius: 4px;
      text-transform: uppercase;
      transition: all 0.2s;
    }

    .typetube-letter-box.locked {
      background: rgba(55, 65, 81, 0.5);
      color: rgba(107, 114, 128, 0.6);
    }

    .typetube-letter-box.learning {
      background: rgba(251, 191, 36, 0.25);
      color: #fbbf24;
      border: 1px solid rgba(251, 191, 36, 0.5);
    }

    .typetube-letter-box.mastered {
      background: rgba(34, 197, 94, 0.25);
      color: #4ade80;
      border: 1px solid rgba(34, 197, 94, 0.5);
    }

    .typetube-letter-box.emphasis {
      background: rgba(139, 92, 246, 0.3);
      color: #c4b5fd;
      border: 2px solid #8b5cf6;
      animation: typetube-emphasis-pulse 2s ease-in-out infinite;
    }

    @keyframes typetube-emphasis-pulse {
      0%, 100% { box-shadow: 0 0 0 0 rgba(139, 92, 246, 0.4); }
      50% { box-shadow: 0 0 8px 2px rgba(139, 92, 246, 0.3); }
    }

    /* Typing display area - centered text */
    .typetube-display-wrapper {
      display: flex;
      flex-direction: column;
      align-items: center;
      gap: 1.5rem;
      max-width: 900px;
      width: 100%;
    }

    .typetube-display {
      font-family: 'Fira Code', 'Consolas', 'Monaco', monospace;
      font-size: 3rem;
      line-height: 1.6;
      padding: 1.5rem 2rem;
      background: rgba(0, 0, 0, 0.4);
      border-radius: 1rem;
      display: flex;
      flex-wrap: wrap;
      gap: 0;
      justify-content: center;
      align-items: center;
      min-height: 5rem;
    }

    .typetube-char {
      display: inline-block;
      padding: 0.05em 0.03em;
      border-radius: 4px;
      transition: all 0.1s ease;
      min-width: 0.55em;
      text-align: center;
    }

    .typetube-char.pending {
      color: rgba(255, 255, 255, 0.35);
    }

    .typetube-char.correct {
      color: #4ade80;
    }

    .typetube-char.corrected {
      color: #fbbf24;
      text-decoration: underline;
      text-decoration-color: #fbbf24;
      text-decoration-thickness: 3px;
    }

    .typetube-char.current {
      background: rgba(139, 92, 246, 0.5);
      color: #fff;
      border-bottom: 4px solid #8b5cf6;
      animation: typetube-pulse 1s ease-in-out infinite;
    }

    .typetube-char.current.incorrect {
      background: rgba(239, 68, 68, 0.6);
      color: #fca5a5;
      border-bottom: 4px solid #ef4444;
      animation: typetube-shake 0.3s ease-in-out;
    }

    .typetube-char.space {
      color: rgba(255, 255, 255, 0.15);
      min-width: 0.7em;
    }

    .typetube-char.space.correct {
      color: rgba(74, 222, 128, 0.3);
    }

    @keyframes typetube-pulse {
      0%, 100% { opacity: 1; }
      50% { opacity: 0.75; }
    }

    @keyframes typetube-shake {
      0%, 100% { transform: translateX(0); }
      25% { transform: translateX(-5px); }
      75% { transform: translateX(5px); }
    }

    /* On-screen keyboard styles - improved */
    .typetube-keyboard-wrapper {
      width: 100%;
      max-width: 700px;
    }

    .typetube-keyboard-toggle {
      display: flex;
      justify-content: flex-end;
      margin-bottom: 0.5rem;
    }

    .typetube-keyboard-toggle button {
      padding: 0.35rem 0.75rem;
      background: rgba(139, 92, 246, 0.15);
      border: 1px solid rgba(139, 92, 246, 0.3);
      border-radius: 0.25rem;
      color: rgba(196, 181, 253, 0.8);
      font-size: 0.85rem;
      cursor: pointer;
      transition: all 0.2s;
    }

    .typetube-keyboard-toggle button:hover {
      background: rgba(139, 92, 246, 0.25);
      border-color: rgba(139, 92, 246, 0.5);
    }

    .typetube-keyboard {
      display: flex;
      flex-direction: column;
      align-items: center;
      gap: 6px;
      padding: 0.75rem;
      background: rgba(0, 0, 0, 0.3);
      border-radius: 0.75rem;
    }

    .typetube-keyboard.hidden {
      display: none;
    }

    .typetube-keyboard-row {
      display: flex;
      gap: 6px;
    }

    /* Proper QWERTY staggering */
    .typetube-keyboard-row.row-0 { margin-left: 0; }
    .typetube-keyboard-row.row-1 { margin-left: 18px; }
    .typetube-keyboard-row.row-2 { margin-left: 30px; }
    .typetube-keyboard-row.row-3 { margin-left: 0; }

    .typetube-key {
      position: relative;
      display: flex;
      align-items: center;
      justify-content: center;
      width: 48px;
      height: 48px;
      border-radius: 6px;
      font-size: 1.25rem;
      font-weight: 700;
      font-family: 'Fira Code', monospace;
      color: white;
      border: none;
      box-shadow: 0 3px 6px rgba(0, 0, 0, 0.3), inset 0 1px 0 rgba(255, 255, 255, 0.15);
      transition: all 0.1s ease;
    }

    .typetube-key.wide {
      width: 220px;
      font-size: 1rem;
    }

    .typetube-key.next {
      transform: scale(1.15);
      box-shadow: 0 0 20px currentColor, 0 0 40px currentColor, 0 4px 8px rgba(0, 0, 0, 0.4);
      z-index: 10;
      border: 2px solid white;
    }

    .typetube-key-dot {
      position: absolute;
      bottom: 6px;
      left: 50%;
      transform: translateX(-50%);
      width: 6px;
      height: 6px;
      background: rgba(255, 255, 255, 0.7);
      border-radius: 50%;
    }

    /* Celebration animation */
    .typetube-celebration {
      position: fixed;
      top: 50%;
      left: 50%;
      transform: translate(-50%, -50%);
      font-size: 4rem;
      font-weight: 700;
      color: #fbbf24;
      text-shadow: 0 0 30px rgba(251, 191, 36, 0.9);
      pointer-events: none;
      z-index: 1000000;
      animation: typetube-celebrate 1.5s ease-out forwards;
    }

    @keyframes typetube-celebrate {
      0% {
        opacity: 0;
        transform: translate(-50%, -50%) scale(0.5);
      }
      30% {
        opacity: 1;
        transform: translate(-50%, -50%) scale(1.2);
      }
      100% {
        opacity: 0;
        transform: translate(-50%, -80%) scale(1);
      }
    }

    /* Power bar pulse on fill */
    .typetube-power-bar-fill.filling {
      animation: typetube-power-pulse 0.3s ease-out;
    }

    @keyframes typetube-power-pulse {
      0%, 100% { transform: scaleY(1); }
      50% { transform: scaleY(1.15); }
    }

    /* Smooth word transitions */
    .typetube-display.transitioning {
      animation: typetube-word-transition 0.3s ease-out;
    }

    @keyframes typetube-word-transition {
      0% {
        opacity: 0.5;
        transform: translateY(10px);
      }
      100% {
        opacity: 1;
        transform: translateY(0);
      }
    }

    /* Help text */
    .typetube-help {
      color: rgba(255, 255, 255, 0.4);
      font-size: 0.75rem;
      text-align: center;
    }

    .typetube-help-key {
      display: inline-block;
      padding: 0.15rem 0.4rem;
      background: rgba(255, 255, 255, 0.1);
      border-radius: 0.2rem;
      font-family: monospace;
      margin: 0 0.2rem;
    }

    /* Mini mode overlay - for optional practice */
    #typetube-mini-overlay {
      position: fixed;
      bottom: 20px;
      left: 50%;
      transform: translateX(-50%);
      background: rgba(20, 20, 30, 0.95);
      border: 2px solid rgba(139, 92, 246, 0.4);
      border-radius: 1rem;
      padding: 1rem 1.5rem;
      z-index: 2147483646;
      display: none;
    }

    #typetube-mini-overlay.visible {
      display: block;
    }

    .typetube-mini-display {
      font-family: 'Fira Code', monospace;
      font-size: 1.5rem;
      color: white;
      text-align: center;
    }

    .typetube-mini-hint {
      font-size: 0.65rem;
      color: rgba(255, 255, 255, 0.4);
      text-align: center;
      margin-top: 0.5rem;
    }
    /* Achievement notification styles */
    .typetube-achievement-notification {
      position: fixed;
      top: 80px;
      right: 20px;
      display: flex;
      align-items: center;
      gap: 12px;
      padding: 16px 20px;
      background: linear-gradient(135deg, rgba(139, 92, 246, 0.95) 0%, rgba(109, 40, 217, 0.95) 100%);
      border: 2px solid rgba(255, 255, 255, 0.3);
      border-radius: 12px;
      box-shadow: 0 8px 32px rgba(139, 92, 246, 0.5), 0 0 0 1px rgba(255, 255, 255, 0.1);
      z-index: 2147483647;
      animation: typetube-achievement-slide-in 0.5s ease-out;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    }

    .typetube-achievement-notification.hiding {
      animation: typetube-achievement-slide-out 0.5s ease-in forwards;
    }

    @keyframes typetube-achievement-slide-in {
      0% { transform: translateX(120%); opacity: 0; }
      100% { transform: translateX(0); opacity: 1; }
    }

    @keyframes typetube-achievement-slide-out {
      0% { transform: translateX(0); opacity: 1; }
      100% { transform: translateX(120%); opacity: 0; }
    }

    .typetube-achievement-icon {
      font-size: 40px;
      animation: typetube-achievement-bounce 0.6s ease-out 0.3s;
    }

    @keyframes typetube-achievement-bounce {
      0%, 100% { transform: scale(1); }
      50% { transform: scale(1.3); }
    }

    .typetube-achievement-content {
      display: flex;
      flex-direction: column;
      gap: 2px;
    }

    .typetube-achievement-title {
      font-size: 11px;
      font-weight: 600;
      color: rgba(255, 255, 255, 0.8);
      text-transform: uppercase;
      letter-spacing: 0.5px;
    }

    .typetube-achievement-name {
      font-size: 18px;
      font-weight: 700;
      color: white;
    }

    .typetube-achievement-desc {
      font-size: 12px;
      color: rgba(255, 255, 255, 0.7);
    }

    /* Letter unlock celebration styles */
    .typetube-letter-unlock-celebration {
      position: fixed;
      top: 0;
      left: 0;
      right: 0;
      bottom: 0;
      display: flex;
      align-items: center;
      justify-content: center;
      background: rgba(0, 0, 0, 0.7);
      z-index: 2147483647;
      animation: typetube-celebration-fade-in 0.3s ease-out;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    }

    .typetube-letter-unlock-celebration.hiding {
      animation: typetube-celebration-fade-out 0.5s ease-in forwards;
    }

    @keyframes typetube-celebration-fade-in {
      0% { opacity: 0; }
      100% { opacity: 1; }
    }

    @keyframes typetube-celebration-fade-out {
      0% { opacity: 1; }
      100% { opacity: 0; }
    }

    .typetube-letter-unlock-message {
      text-align: center;
      z-index: 2;
    }

    .typetube-letter-unlock-label {
      font-size: 24px;
      font-weight: 700;
      color: #fbbf24;
      text-shadow: 0 0 20px rgba(251, 191, 36, 0.5);
      margin-bottom: 16px;
      animation: typetube-unlock-label-pulse 1s ease-in-out infinite;
    }

    @keyframes typetube-unlock-label-pulse {
      0%, 100% { transform: scale(1); }
      50% { transform: scale(1.05); }
    }

    .typetube-letter-unlock-letter {
      font-size: 120px;
      font-weight: 900;
      color: white;
      text-shadow: 0 0 40px rgba(139, 92, 246, 0.8), 0 0 80px rgba(139, 92, 246, 0.4);
      animation: typetube-unlock-letter-pop 0.6s ease-out;
    }

    @keyframes typetube-unlock-letter-pop {
      0% { transform: scale(0); opacity: 0; }
      60% { transform: scale(1.2); }
      100% { transform: scale(1); opacity: 1; }
    }

    /* Confetti styles */
    .typetube-confetti-container {
      position: fixed;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      pointer-events: none;
      overflow: hidden;
      z-index: 1;
    }

    .typetube-confetti {
      position: absolute;
      top: -10px;
      width: 10px;
      height: 10px;
      border-radius: 2px;
      animation: typetube-confetti-fall linear forwards;
    }

    @keyframes typetube-confetti-fall {
      0% {
        transform: translateY(0) rotate(0deg);
        opacity: 1;
      }
      100% {
        transform: translateY(100vh) rotate(720deg);
        opacity: 0;
      }
    }
  `;
    document.head.appendChild(style);
  }
  function renderKeyboard() {
    const session = state.session;
    const nextChar = session.currentIndex < session.characters.length ? session.characters[session.currentIndex].char.toLowerCase() : "";
    let html = "";
    KEYBOARD_LAYOUT.forEach((row, rowIndex) => {
      html += `<div class="typetube-keyboard-row row-${rowIndex}">`;
      row.forEach((keyInfo) => {
        const fingerColor = FINGER_COLORS[keyInfo.finger];
        const isNext = keyInfo.key.toLowerCase() === nextChar;
        const wideClass = keyInfo.width && keyInfo.width > 2 ? "wide" : "";
        const nextClass = isNext ? "next" : "";
        const homeDot = keyInfo.isHomeKey ? '<span class="typetube-key-dot"></span>' : "";
        html += `
        <div class="typetube-key ${wideClass} ${nextClass}" style="background: ${fingerColor};">
          ${keyInfo.display}
          ${homeDot}
        </div>
      `;
      });
      html += "</div>";
    });
    return html;
  }
  function renderLetterProgress() {
    const emphasisLetter = letterProgressState ? getEmphasisLetter(letterProgressState) : null;
    const unlockedSet = letterProgressState?.unlockedLetters || /* @__PURE__ */ new Set(["e", "n", "i", "a", "r", "l", "t", "o"]);
    let html = '<div class="typetube-letter-grid-keybr">';
    LETTER_ORDER.forEach((letter) => {
      const isUnlocked = unlockedSet.has(letter);
      const isEmphasis = letter === emphasisLetter;
      let status = "locked";
      if (isEmphasis) {
        status = "emphasis";
      } else if (isUnlocked) {
        const stats = letterProgressState?.letterStats.get(letter);
        status = stats?.isMastered ? "mastered" : "learning";
      }
      html += `<div class="typetube-letter-box ${status}">${letter}</div>`;
    });
    html += "</div>";
    return html;
  }
  function toggleKeyboard() {
    state.keyboardVisible = !state.keyboardVisible;
    const keyboard = document.getElementById("typetube-keyboard");
    const toggleBtn = document.getElementById("typetube-keyboard-toggle");
    if (keyboard) {
      keyboard.classList.toggle("hidden", !state.keyboardVisible);
    }
    if (toggleBtn) {
      toggleBtn.textContent = state.keyboardVisible ? "Hide Keyboard" : "Show Keyboard";
    }
  }
  function showCelebration(milestone) {
    const celebration = document.createElement("div");
    celebration.className = "typetube-celebration";
    let message = "";
    let emoji = "";
    if (milestone >= 100) {
      message = "LEGENDARY!";
      emoji = "\u{1F3C6}";
    } else if (milestone >= 50) {
      message = "ON FIRE!";
      emoji = "\u{1F525}";
    } else if (milestone >= 25) {
      message = "AWESOME!";
      emoji = "\u2B50";
    } else if (milestone >= 10) {
      message = "GREAT!";
      emoji = "\u2728";
    }
    celebration.innerHTML = `${emoji} ${message} ${emoji}`;
    document.body.appendChild(celebration);
    setTimeout(() => {
      celebration.remove();
    }, 1500);
  }
  function showLetterUnlockCelebration(letter) {
    const celebration = document.createElement("div");
    celebration.className = "typetube-letter-unlock-celebration";
    const confettiContainer = document.createElement("div");
    confettiContainer.className = "typetube-confetti-container";
    const colors = ["#fbbf24", "#8b5cf6", "#10b981", "#ec4899", "#3b82f6"];
    for (let i = 0; i < 50; i++) {
      const confetti = document.createElement("div");
      confetti.className = "typetube-confetti";
      confetti.style.left = `${Math.random() * 100}%`;
      confetti.style.backgroundColor = colors[Math.floor(Math.random() * colors.length)];
      confetti.style.animationDelay = `${Math.random() * 0.5}s`;
      confetti.style.animationDuration = `${1.5 + Math.random()}s`;
      confettiContainer.appendChild(confetti);
    }
    const message = document.createElement("div");
    message.className = "typetube-letter-unlock-message";
    message.innerHTML = `
    <div class="typetube-letter-unlock-label">NEW LETTER UNLOCKED!</div>
    <div class="typetube-letter-unlock-letter">${letter.toUpperCase()}</div>
  `;
    celebration.appendChild(confettiContainer);
    celebration.appendChild(message);
    document.body.appendChild(celebration);
    setTimeout(() => {
      celebration.classList.add("hiding");
      setTimeout(() => {
        celebration.remove();
      }, 500);
    }, 2500);
  }
  var ACHIEVEMENTS = [
    { id: "first_drill", name: "First Steps", description: "Complete your first drill", icon: "\u{1F3AF}" },
    { id: "speed_demon", name: "Speed Demon", description: "Reach 30 WPM", icon: "\u26A1" },
    { id: "accuracy_star", name: "Accuracy Star", description: "Get 95% accuracy", icon: "\u{1F3AF}" },
    { id: "streak_master", name: "Streak Master", description: "Get a 25 character streak", icon: "\u{1F525}" },
    { id: "letter_learner", name: "Letter Learner", description: "Unlock 15 letters", icon: "\u{1F4DA}" },
    { id: "alphabet_hero", name: "Alphabet Hero", description: "Unlock all 26 letters", icon: "\u{1F3C6}" },
    { id: "practice_pro", name: "Practice Pro", description: "Complete 10 drills in a session", icon: "\u{1F4AA}" },
    { id: "perfectionist", name: "Perfectionist", description: "Get 100% accuracy on a drill", icon: "\u2728" }
  ];
  var ACHIEVEMENTS_STORAGE_KEY = "typetube_achievements";
  async function getUnlockedAchievements() {
    return new Promise((resolve) => {
      chrome.storage.local.get([ACHIEVEMENTS_STORAGE_KEY], (result) => {
        const unlocked = result[ACHIEVEMENTS_STORAGE_KEY] || [];
        resolve(new Set(unlocked));
      });
    });
  }
  async function saveUnlockedAchievements(unlocked) {
    return new Promise((resolve) => {
      chrome.storage.local.set({ [ACHIEVEMENTS_STORAGE_KEY]: Array.from(unlocked) }, resolve);
    });
  }
  async function checkAchievementsForDrill(stats) {
    const unlocked = await getUnlockedAchievements();
    const newlyUnlocked = [];
    const checks = [
      { id: "first_drill", condition: stats.drillsCompleted >= 1 },
      { id: "speed_demon", condition: stats.wpm >= 30 },
      { id: "accuracy_star", condition: stats.accuracy >= 95 },
      { id: "streak_master", condition: stats.maxStreak >= 25 },
      { id: "letter_learner", condition: stats.lettersUnlocked >= 15 },
      { id: "alphabet_hero", condition: stats.lettersUnlocked >= 26 },
      { id: "practice_pro", condition: stats.drillsCompleted >= 10 },
      { id: "perfectionist", condition: stats.accuracy >= 100 }
    ];
    for (const check of checks) {
      if (check.condition && !unlocked.has(check.id)) {
        const achievement = ACHIEVEMENTS.find((a) => a.id === check.id);
        if (achievement) {
          unlocked.add(check.id);
          newlyUnlocked.push(achievement);
        }
      }
    }
    if (newlyUnlocked.length > 0) {
      await saveUnlockedAchievements(unlocked);
    }
    return newlyUnlocked;
  }
  function showAchievementNotification(achievement) {
    const notification = document.createElement("div");
    notification.className = "typetube-achievement-notification";
    notification.innerHTML = `
    <div class="typetube-achievement-icon">${achievement.icon}</div>
    <div class="typetube-achievement-content">
      <div class="typetube-achievement-title">Achievement Unlocked!</div>
      <div class="typetube-achievement-name">${achievement.name}</div>
      <div class="typetube-achievement-desc">${achievement.description}</div>
    </div>
  `;
    document.body.appendChild(notification);
    setTimeout(() => {
      notification.classList.add("hiding");
      setTimeout(() => {
        notification.remove();
      }, 500);
    }, 3e3);
  }
  function checkStreakCelebration() {
    const streak = state.session.streak;
    const milestones = [10, 25, 50, 100];
    for (const milestone of milestones) {
      if (streak === milestone && state.lastCelebratedStreak < milestone) {
        state.lastCelebratedStreak = milestone;
        showCelebration(milestone);
        break;
      }
    }
  }
  function formatChar(char) {
    return char === " " ? "\xB7" : char;
  }
  function getCharClass(charState, index) {
    const session = state.session;
    const classes = ["typetube-char"];
    if (index < session.currentIndex) {
      if (charState.status === "correct") {
        classes.push("correct");
      } else if (charState.status === "corrected") {
        classes.push("corrected");
      }
    } else if (index === session.currentIndex) {
      classes.push("current");
      if (session.hasIncorrectChar) {
        classes.push("incorrect");
      }
    } else {
      classes.push("pending");
    }
    if (charState.char === " ") {
      classes.push("space");
    }
    return classes.join(" ");
  }
  function getDisplayChar(charState, index) {
    const session = state.session;
    if (index === session.currentIndex && session.hasIncorrectChar && charState.typedChar) {
      return charState.typedChar;
    }
    return formatChar(charState.char);
  }
  function showOverlay() {
    if (overlay) {
      console.log("[TypeTube] Overlay already exists, not recreating");
      return;
    }
    console.log("[TypeTube] Creating typing overlay");
    if (powerBarElement) {
      powerBarElement.classList.add("hidden");
    }
    const powerPercentage = Math.round(powerBarState.currentPower / powerBarState.maxPower * 100);
    overlay = document.createElement("div");
    overlay.id = "typetube-overlay";
    overlay.innerHTML = `
    <!-- Main typing area -->
    <div class="typetube-main">
      <div class="typetube-display-wrapper">
        <div class="typetube-display" id="typetube-display"></div>
        <div class="typetube-keyboard-wrapper">
          <div class="typetube-keyboard-toggle">
            <button id="typetube-keyboard-toggle">${state.keyboardVisible ? "Hide Keyboard" : "Show Keyboard"}</button>
          </div>
          <div class="typetube-keyboard ${state.keyboardVisible ? "" : "hidden"}" id="typetube-keyboard">
            ${renderKeyboard()}
          </div>
        </div>
        <div class="typetube-help">
          Press <span class="typetube-help-key">Backspace</span> to fix errors
        </div>
      </div>
    </div>
    <!-- Stats sidebar -->
    <div class="typetube-sidebar">
      <div class="typetube-sidebar-section">
        <div class="typetube-sidebar-label">Speed</div>
        <div class="typetube-stat-row wpm">
          <span class="typetube-stat-value-large" id="typetube-wpm">0</span>
          <span class="typetube-stat-unit">wpm</span>
        </div>
      </div>
      <div class="typetube-sidebar-section">
        <div class="typetube-sidebar-label">Accuracy</div>
        <div class="typetube-stat-row accuracy">
          <span class="typetube-stat-value-large" id="typetube-accuracy">100</span>
          <span class="typetube-stat-unit">%</span>
        </div>
      </div>
      <div class="typetube-sidebar-section">
        <div class="typetube-sidebar-label">Streak</div>
        <div class="typetube-stat-row streak">
          <span class="typetube-stat-value-large" id="typetube-streak">0</span>
          <span id="typetube-streak-icon"></span>
        </div>
        <div class="typetube-streak-badge" id="typetube-multiplier" style="display:none;">
          <span>1x bonus</span>
        </div>
      </div>
      <div class="typetube-sidebar-section">
        <div class="typetube-sidebar-label">Letters</div>
        ${renderLetterProgress()}
      </div>
      <div class="typetube-sidebar-section typetube-sidebar-power">
        <div class="typetube-sidebar-label">Power</div>
        <div class="typetube-power-bar-container">
          <div class="typetube-power-bar-fill power-${getPowerBarColor(powerPercentage)}" id="typetube-overlay-power-fill" style="width: ${powerPercentage}%"></div>
          <div class="typetube-power-bar-text" id="typetube-overlay-power-text">${powerPercentage}%</div>
        </div>
      </div>
    </div>
  `;
    document.body.appendChild(overlay);
    renderDisplay();
    updateStats();
    const toggleBtn = document.getElementById("typetube-keyboard-toggle");
    if (toggleBtn) {
      toggleBtn.addEventListener("click", toggleKeyboard);
    }
    document.addEventListener("keydown", handleKeyDown, true);
    if (statsUpdateInterval) {
      clearInterval(statsUpdateInterval);
    }
    statsUpdateInterval = window.setInterval(updateStats, 200);
  }
  function handleKeyDown(e) {
    if (!state.isActive || !overlay) {
      return;
    }
    if (e.key !== "Tab") {
      e.preventDefault();
      e.stopPropagation();
      e.stopImmediatePropagation();
    }
    if (e.key === "Backspace") {
      if (handleBackspace()) {
        renderDisplay();
        updateStats();
      }
      return;
    }
    if (e.key.length !== 1 || e.ctrlKey || e.metaKey || e.altKey) {
      return;
    }
    const result = handleKeyPress(e.key);
    if (result) {
      renderDisplay();
      updateStats();
      checkStreakCelebration();
    }
    if (state.session.isComplete) {
      if (powerBarState.isFull) {
        console.log("[TypeTube] Power bar full AND text complete - challenge complete!");
        completeChallenge();
      } else {
        console.log("[TypeTube] Text complete but bar not full - generating new words");
        generateNewWords();
      }
    }
  }
  function generateNewWords() {
    const challengeText = generateChallenge();
    resetSession(challengeText);
    const displayContainer = document.getElementById("typetube-display");
    if (displayContainer) {
      displayContainer.classList.add("transitioning");
      setTimeout(() => displayContainer.classList.remove("transitioning"), 300);
    }
    renderDisplay();
    updateStats();
    console.log("[TypeTube] New challenge generated:", challengeText);
  }
  function renderDisplay() {
    const container = document.getElementById("typetube-display");
    if (!container) return;
    let html = "";
    for (let i = 0; i < state.session.characters.length; i++) {
      const charState = state.session.characters[i];
      const className = getCharClass(charState, i);
      const displayChar = getDisplayChar(charState, i);
      html += `<span class="${className}">${displayChar === " " ? "&nbsp;" : displayChar}</span>`;
    }
    container.innerHTML = html;
    const keyboardContainer = document.getElementById("typetube-keyboard");
    if (keyboardContainer) {
      keyboardContainer.innerHTML = renderKeyboard();
    }
  }
  function updateOverlayPowerBar() {
    const fillElement = document.getElementById("typetube-overlay-power-fill");
    const textElement = document.getElementById("typetube-overlay-power-text");
    if (fillElement) {
      const percentage = Math.round(powerBarState.currentPower / powerBarState.maxPower * 100);
      const color = getPowerBarColor(percentage);
      fillElement.style.width = `${percentage}%`;
      fillElement.classList.remove("power-green", "power-yellow", "power-red", "power-critical");
      fillElement.classList.add(`power-${color}`);
    }
    if (textElement) {
      const percentage = Math.round(powerBarState.currentPower / powerBarState.maxPower * 100);
      textElement.textContent = `${percentage}%`;
    }
  }
  function updateStats() {
    const wpmEl = document.getElementById("typetube-wpm");
    const accuracyEl = document.getElementById("typetube-accuracy");
    const streakEl = document.getElementById("typetube-streak");
    const streakIconEl = document.getElementById("typetube-streak-icon");
    const multiplierEl = document.getElementById("typetube-multiplier");
    if (wpmEl) {
      wpmEl.textContent = String(getWPM());
    }
    if (accuracyEl) {
      accuracyEl.textContent = String(getSessionAccuracy());
    }
    if (streakEl) {
      streakEl.textContent = String(state.session.streak);
    }
    if (streakIconEl) {
      const streak = state.session.streak;
      if (streak >= 50) {
        streakIconEl.textContent = " \u{1F525}";
      } else if (streak >= 20) {
        streakIconEl.textContent = " \u2B50";
      } else if (streak >= 10) {
        streakIconEl.textContent = " \u26A1";
      } else if (streak > 0) {
        streakIconEl.textContent = " \u2728";
      } else {
        streakIconEl.textContent = "";
      }
    }
    if (multiplierEl) {
      const multiplier = getStreakMultiplier(state.session.streak);
      if (multiplier > 1) {
        multiplierEl.style.display = "block";
        multiplierEl.innerHTML = `<span>${multiplier}x bonus</span>`;
      } else {
        multiplierEl.style.display = "none";
      }
    }
  }
  function completeChallenge() {
    console.log("[TypeTube] Power bar full - challenge completed!");
    if (statsUpdateInterval) {
      clearInterval(statsUpdateInterval);
      statsUpdateInterval = null;
    }
    const wpm = getWPM();
    const accuracy = getSessionAccuracy();
    const session = state.session;
    updateEstimatedWPM(wpm);
    sessionDrillsCompleted++;
    let lettersUnlocked = 8;
    if (letterProgressState) {
      lettersUnlocked = letterProgressState.unlockedLetters.size;
    }
    if (letterProgressState) {
      const unlockedNew = checkForNewLetterUnlock(letterProgressState);
      if (unlockedNew) {
        const summary = getProgressSummary(letterProgressState);
        console.log(
          `[TypeTube] New letter unlocked! Progress: ${summary.unlockedCount}/${summary.totalLetters} letters`
        );
        const newLetter = letterProgressState.currentEmphasisLetter;
        if (newLetter) {
          showLetterUnlockCelebration(newLetter);
        }
        lettersUnlocked = letterProgressState.unlockedLetters.size;
      }
      saveLetterProgress(letterProgressState).catch((e) => {
        console.error("[TypeTube] Error saving letter progress:", e);
      });
    }
    checkAchievementsForDrill({
      wpm,
      accuracy,
      maxStreak: session.maxStreak,
      drillsCompleted: sessionDrillsCompleted,
      lettersUnlocked
    }).then((newAchievements) => {
      newAchievements.forEach((achievement, index) => {
        setTimeout(() => {
          showAchievementNotification(achievement);
        }, index * 500);
      });
    }).catch((e) => {
      console.error("[TypeTube] Error checking achievements:", e);
    });
    saveStats({
      wpm,
      accuracy,
      chars: session.correctCount,
      errors: session.errorCount,
      corrected: session.correctedCount,
      maxStreak: session.maxStreak
    });
    hideOverlay();
    if (state.videoElement) {
      state.videoElement.play();
      startDrain();
    }
    state.isActive = false;
  }
  function updateEstimatedWPM(newWPM) {
    powerBarState.estimatedWPM = powerBarState.estimatedWPM * 0.7 + newWPM * 0.3;
    powerBarState.drainRate = calculateDrainRate(powerBarState.estimatedWPM, powerBarState.targetTypingRatio);
    powerBarState.baseFillPerChar = calculateBaseFillPerChar(
      powerBarState.drainRate,
      powerBarState.estimatedWPM,
      powerBarState.targetTypingRatio
    );
    chrome.storage.sync.set({ estimatedWPM: powerBarState.estimatedWPM }).catch((e) => {
      console.error("[TypeTube] Error saving estimated WPM:", e);
    });
  }
  function hideOverlay() {
    console.log("[TypeTube] Hiding overlay");
    document.removeEventListener("keydown", handleKeyDown, true);
    if (overlay) {
      overlay.remove();
      overlay = null;
    }
    if (powerBarElement) {
      powerBarElement.classList.remove("hidden");
    }
  }
  async function saveStats(stats) {
    try {
      if (!currentUserId) {
        console.warn("[TypeTube] No current user, cannot save stats");
        return;
      }
      const now = Date.now();
      const videoId = getVideoId();
      const startTime = state.session.startTime || now;
      const statsKey = `typingStats_${currentUserId}`;
      const existing = await chrome.storage.local.get([statsKey]);
      const allStats = existing[statsKey] || [];
      allStats.push({
        ...stats,
        timestamp: now,
        startTime,
        endTime: now,
        videoId,
        videoTitle: currentVideoTitle || null,
        brainrotFactor: currentBrainrotFactor
      });
      await chrome.storage.local.set({ [statsKey]: allStats });
      console.log("[TypeTube] Stats saved to local storage for user:", currentUserId);
      try {
        await chrome.runtime.sendMessage({
          type: "QUEUE_SESSION",
          session: {
            user_id: currentUserId,
            video_id: videoId,
            video_title: currentVideoTitle || null,
            started_at: startTime,
            ended_at: now,
            total_characters: stats.chars,
            correct_characters: stats.chars - stats.errors,
            errors: stats.errors,
            corrected: stats.corrected,
            max_streak: stats.maxStreak,
            wpm: stats.wpm,
            accuracy: stats.accuracy,
            brainrot_factor: currentBrainrotFactor
          }
        });
        console.log("[TypeTube] Session queued for sync to API");
      } catch (queueError) {
        console.error("[TypeTube] Error queuing session for sync:", queueError);
      }
    } catch (e) {
      console.error("[TypeTube] Error saving stats:", e);
    }
  }
  function getVideoId() {
    const url = new URL(window.location.href);
    return url.searchParams.get("v");
  }
  var miniSession = null;
  function showMiniOverlay() {
    if (!miniModeEnabled || miniOverlay || state.isActive) return;
    const challengeText = generateChallenge();
    miniSession = {
      targetText: challengeText,
      characters: challengeText.split("").map((char) => ({
        char,
        status: "pending"
      })),
      currentIndex: 0,
      streak: 0,
      maxStreak: 0,
      correctCount: 0,
      errorCount: 0,
      correctedCount: 0,
      startTime: null,
      isComplete: false,
      hasIncorrectChar: false,
      letterTimings: /* @__PURE__ */ new Map()
    };
    miniOverlay = document.createElement("div");
    miniOverlay.id = "typetube-mini-overlay";
    miniOverlay.classList.add("visible");
    renderMiniDisplay();
    document.body.appendChild(miniOverlay);
    document.addEventListener("keydown", handleMiniKeyDown, true);
    document.addEventListener("keyup", handleMiniKeyUp, true);
  }
  function hideMiniOverlay() {
    if (miniOverlay) {
      document.removeEventListener("keydown", handleMiniKeyDown, true);
      document.removeEventListener("keyup", handleMiniKeyUp, true);
      miniOverlay.remove();
      miniOverlay = null;
      miniSession = null;
    }
  }
  function renderMiniDisplay() {
    if (!miniOverlay || !miniSession) return;
    let charHtml = "";
    for (let i = 0; i < miniSession.characters.length; i++) {
      const charState = miniSession.characters[i];
      let className = "typetube-char";
      if (i < miniSession.currentIndex) {
        className += charState.status === "correct" ? " correct" : " corrected";
      } else if (i === miniSession.currentIndex) {
        className += " current";
        if (miniSession.hasIncorrectChar) {
          className += " incorrect";
        }
      } else {
        className += " pending";
      }
      if (charState.char === " ") {
        className += " space";
      }
      const displayChar = charState.char === " " ? "\xB7" : charState.char;
      charHtml += `<span class="${className}">${displayChar}</span>`;
    }
    miniOverlay.innerHTML = `
    <div class="typetube-mini-display">${charHtml}</div>
    <div class="typetube-mini-hint">Type to gain power \u2022 Press Esc to hide</div>
  `;
  }
  function handleMiniKeyDown(e) {
    if (!miniOverlay || !miniSession || state.isActive) return;
    const target = e.target;
    if (target.tagName === "INPUT" || target.tagName === "TEXTAREA" || target.isContentEditable) {
      return;
    }
    if (e.key === "Escape") {
      e.preventDefault();
      e.stopPropagation();
      hideMiniOverlay();
      return;
    }
    const isSingleChar = e.key.length === 1;
    const hasModifier = e.ctrlKey || e.metaKey || e.altKey;
    if (isSingleChar && !hasModifier) {
      e.preventDefault();
      e.stopPropagation();
      e.stopImmediatePropagation();
    }
    if (e.key === "Backspace") {
      e.preventDefault();
      e.stopPropagation();
      e.stopImmediatePropagation();
      if (miniSession.hasIncorrectChar && miniSession.currentIndex < miniSession.characters.length) {
        const currentChar2 = miniSession.characters[miniSession.currentIndex];
        currentChar2.status = "pending";
        currentChar2.typedChar = void 0;
        miniSession.hasIncorrectChar = false;
        renderMiniDisplay();
      }
      return;
    }
    if (miniSession.hasIncorrectChar) return;
    if (!isSingleChar || hasModifier) return;
    if (miniSession.currentIndex >= miniSession.characters.length) return;
    const currentChar = miniSession.characters[miniSession.currentIndex];
    const expectedChar = currentChar.char;
    if (e.key === expectedChar) {
      currentChar.status = "correct";
      currentChar.typedChar = e.key;
      miniSession.correctCount++;
      miniSession.currentIndex++;
      miniSession.streak++;
      addPowerForCorrectChar();
      if (miniSession.currentIndex >= miniSession.characters.length) {
        const newText = generateChallenge();
        miniSession.targetText = newText;
        miniSession.characters = newText.split("").map((char) => ({
          char,
          status: "pending"
        }));
        miniSession.currentIndex = 0;
        miniSession.hasIncorrectChar = false;
      }
    } else {
      currentChar.status = "incorrect";
      currentChar.typedChar = e.key;
      miniSession.errorCount++;
      miniSession.streak = 0;
      miniSession.hasIncorrectChar = true;
      applyErrorPenalty();
    }
    renderMiniDisplay();
  }
  function handleMiniKeyUp(e) {
    if (!miniOverlay || !miniSession || state.isActive) return;
    const target = e.target;
    if (target.tagName === "INPUT" || target.tagName === "TEXTAREA" || target.isContentEditable) {
      return;
    }
    const isSingleChar = e.key.length === 1;
    const hasModifier = e.ctrlKey || e.metaKey || e.altKey;
    if (isSingleChar && !hasModifier) {
      e.preventDefault();
      e.stopPropagation();
      e.stopImmediatePropagation();
    }
    if (e.key === "Backspace") {
      e.preventDefault();
      e.stopPropagation();
      e.stopImmediatePropagation();
    }
  }
  if (document.readyState === "loading") {
    document.addEventListener("DOMContentLoaded", init);
  } else {
    init();
  }
})();
