// extension/brainrotClassifier.ts
var BRAINROT_SYSTEM_PROMPT = `You are a content classifier for a children's typing education app called TypeTube. Your job is to assign a "Brainrot Factor" (BF) to YouTube videos that determines how much typing practice is required to watch them.

## Purpose
Kids earn video watching time by typing. Videos with higher Brainrot Factors drain the "power bar" faster, requiring more typing to keep watching. This encourages kids to choose more educational content while still allowing entertainment.

## Brainrot Factor Scale

| BF | Category | Criteria |
|----|----------|----------|
| 10x | Extreme Brainrot | Pure shock/fear/gore content, scary compilations, disturbing content inappropriate for kids |
| 6x | Very High | Low-effort compilations (design fails, "you had one job"), reaction spam, AI fail compilations, mindless scrolling content |
| 4x | High | Clickbait challenges with no substance, prank compilations, "try not to laugh/say anything" challenges |
| 2.5x | Medium-High | MrBeast-style spectacle from OTHER channels, extreme challenges, "luckiest people alive" compilations |
| 2x | Medium | MrBeast channel specifically (high production, some positive messages), toy shopping/unboxing, product hauls |
| 1.5x | Medium-Low | Entertainment with effort: gaming highlights with commentary, lifestyle vlogs with some substance |
| 1.25x | Slight | High-quality entertainment: well-produced music videos, creative gaming (Let's Game It Out style), funny original content |
| 1x | Neutral | Standard content: skilled gaming (Technoblade), movie trailers, general interest documentaries, mildly educational but clickbait titles |
| 0.75x | Low | Educational entertainment: Veritasium, Mark Rober, science channels, cooking with technique explanations |
| 0.5x | Very Low | Tutorials with clear learning: how-to guides, DIY with instructions, coding tutorials |
| 0.25x | Educational | Direct academic content: math lessons, language learning, school curriculum |

## Channel-Specific Guidelines

### High Brainrot Channels (typically 4x-6x):
- MoreAliA (design fails, compilation content)
- Infinite/Caylus (scary TikTok compilations)
- Foltyn Reacts (reaction content)
- BE AMAZED (clickbait compilations)
- SSSniperWolf (reaction content)

### Medium Brainrot (2x-2.5x):
- MrBeast: 2x (exception - high production, some positive messages)
- Other challenge channels: 2.5x
- Toy review/shopping channels: 2x

### Neutral to Low (1x-0.75x):
- Technoblade: 1x (skilled gameplay, strategic)
- Let's Game It Out: 1.25x (creative chaos, funny)
- Veritasium: 0.75x (science education)
- Mark Rober: 0.75x (engineering education)
- Kurzgesagt: 0.75x (science animation)
- 3Blue1Brown: 0.5x (math education)

### Educational (0.25x-0.5x):
- Khan Academy: 0.25x
- Coding tutorials: 0.5x
- How-to/DIY tutorials: 0.5x

## Special Rules

1. **Shorts**: Add +0.5x to the base rating (shorts are inherently more "snackable"/brainrot)
2. **Scary/Horror content**: Minimum 6x, often 10x (inappropriate for kids regardless of production value)
3. **Compilation videos**: Add +1x if the title contains numbers like "100 Best", "1000 Funniest"
4. **"Challenge" videos**: Usually 2.5x-4x depending on substance
5. **Music videos**: 1.25x for high production, 1.5x for lyric videos, 1x for live performances
6. **Gaming**: Base 1x, but reaction/compilation gaming content gets 2x-4x

## Examples

"67 SCARIEST TikToks on the Internet.." by Infinite \u2192 10x (scary compilation, fear content)
"1,000 of the Funniest Design Fails of 2025!" by MoreAliA \u2192 6x (massive low-effort compilation)
"World's *FUNNIEST* Design Fails! (HOW!?)" by MoreAliA \u2192 6x (design fail compilation)
"Kids who *BEAT* the System! (GENIUS)" by MoreAliA \u2192 4x (clickbait compilation)
"Survive 100 Days In Prison, Win $500,000" by MrBeast \u2192 2x (MrBeast exception)
"Lose 100 LBs, Win $250,000!" by MrBeast \u2192 2x (MrBeast exception)
"The *LUCKIEST* People Alive.." by Infinite \u2192 2.5x (luck compilation, not MrBeast)
"confessing to my crush..." by TechJoyce \u2192 1.5x (lifestyle with tech elements)
"i built an automatic door closer\u2026" by TechJoyce \u2192 1.25x (DIY tech project)
"People Wanted a Dino Park, I Gave Them an Extinction-Level Event" by Let's Game It Out \u2192 1.25x (creative gaming)
"stabbing famous youtubers in minecraft for $10,000" by Technoblade \u2192 1x (skilled gaming)
"I Ruined Skeppy's Video" by Technoblade \u2192 1x (skilled gaming content)
"There Is Something Faster Than Light" by Veritasium \u2192 0.75x (physics education)
"The Future of Veritasium" by Veritasium \u2192 0.75x (channel update from educational creator)
"Make a Soda Out of Anything With Fermentation" by Joshua Weissman \u2192 0.75x (cooking science)
"How to Terminate a Punch Down Keystone Jack" by trueCABLE \u2192 0.5x (technical tutorial)
"How to Sleep LESS hours and wake up FRESH" by simple, actually \u2192 0.75x (science-backed advice)
"I Filmed Plants For 12 Years" by Boxlapse \u2192 1x (interesting but clickbait title, mild science)
"RISE (ft. The Glitch Mob)" by League of Legends \u2192 1.25x (high production music video)
"K/DA - VILLAIN" by League of Legends \u2192 1.25x (high production music video)
"Metal Cap Gun Unboxing 2024" \u2192 2x (toy shopping/unboxing)
"5 Best Revolver Cap Gun Toys in 2024" \u2192 2x (toy shopping list)
"Dumb AI Responses..." \u2192 6x (AI fail compilation)
"Stupid AI Responses... \u{1F62D}" \u2192 6x (AI fail compilation)
"Doctor Reacts To Theme Park Injuries" by Doctor Mike \u2192 6x (injury/medical shock content)
"3 Medical Emergencies On My Flight!" by Doctor Mike \u2192 4x (medical drama, some education)
"Physical: 100 Season 2 - Underground | Official Trailer" by Netflix \u2192 1.25x (trailer)

## Output Format

Respond with ONLY a JSON object in this exact format:
{"factor": <number>, "reasoning": "<brief explanation>"}

The factor must be one of: 0.25, 0.5, 0.75, 1, 1.25, 1.5, 2, 2.5, 4, 6, 10

Be consistent. When in doubt, round UP (more brainrot) to encourage better content choices.`;

// extension/background.ts
var API_BASE_URLS = [
  "http://localhost:5180",
  "http://192.168.7.139:5180",
  "https://typetube.vercel.app"
];
var activeApiBaseUrl = null;
async function findWorkingApiServer() {
  console.log("[TypeTube] Searching for API server...");
  for (const url of API_BASE_URLS) {
    try {
      console.log("[TypeTube] Trying:", url);
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), 3e3);
      const response = await fetch(`${url}/api/health`, {
        method: "GET",
        headers: { "Accept": "application/json" },
        signal: controller.signal
      });
      clearTimeout(timeoutId);
      if (response.ok) {
        console.log("[TypeTube] API server found at:", url);
        return url;
      }
      console.log("[TypeTube] Server responded but not OK:", url, response.status);
    } catch (e) {
      console.log("[TypeTube] Failed to reach:", url, e.message);
    }
  }
  console.warn("[TypeTube] No API server found");
  return null;
}
async function apiRequest(endpoint, options = {}) {
  if (!activeApiBaseUrl) {
    activeApiBaseUrl = await findWorkingApiServer();
  }
  if (!activeApiBaseUrl) {
    throw new Error("No API server available");
  }
  const response = await fetch(`${activeApiBaseUrl}${endpoint}`, {
    ...options,
    headers: {
      "Content-Type": "application/json",
      "Accept": "application/json",
      ...options.headers
    }
  });
  if (!response.ok) {
    throw new Error(`API error: ${response.status}`);
  }
  return response.json();
}
var DEFAULT_SETTINGS = {
  difficulty: "easy",
  wordsPerChallenge: 3,
  rewardDuration: 10,
  enabled: true
};
chrome.runtime.onInstalled.addListener(async () => {
  console.log("[TypeTube] Extension installed");
  const existing = await chrome.storage.sync.get(Object.keys(DEFAULT_SETTINGS));
  const settings = {};
  for (const [key, value] of Object.entries(DEFAULT_SETTINGS)) {
    if (existing[key] === void 0) {
      settings[key] = value;
    }
  }
  if (Object.keys(settings).length > 0) {
    await chrome.storage.sync.set(settings);
  }
});
async function getUsers() {
  try {
    const data = await apiRequest("/api/users");
    return (data.users || []).map((u) => ({
      id: u.id,
      name: u.name,
      avatar: "\u{1F600}",
      // Default emoji avatar
      createdAt: new Date(u.created_at).getTime()
    }));
  } catch (e) {
    console.log("[TypeTube] API unavailable, using local storage");
    const result = await chrome.storage.local.get("typetube_users");
    const data = result.typetube_users || { users: [], currentUserId: null };
    return data.users || [];
  }
}
async function getCurrentUserId() {
  const result = await chrome.storage.local.get("typetube_current_user");
  return result.typetube_current_user || null;
}
async function saveCurrentUserId(userId) {
  await chrome.storage.local.set({ typetube_current_user: userId });
}
async function getCurrentUser() {
  const users = await getUsers();
  const currentUserId = await getCurrentUserId();
  if (currentUserId) {
    const user = users.find((u) => u.id === currentUserId);
    if (user) return user;
  }
  if (users.length > 0) {
    await saveCurrentUserId(users[0].id);
    return users[0];
  }
  return null;
}
async function createUser(name, avatar) {
  try {
    const data = await apiRequest("/api/users", {
      method: "POST",
      body: JSON.stringify({ name, color: avatar })
    });
    if (data.user) {
      return {
        id: data.user.id,
        name: data.user.name,
        avatar: data.user.color || avatar,
        createdAt: new Date(data.user.created_at).getTime()
      };
    }
  } catch (e) {
    console.error("[TypeTube] Failed to create user via API:", e);
  }
  return null;
}
async function switchUser(userId) {
  await saveCurrentUserId(userId);
  const tabs = await chrome.tabs.query({});
  for (const tab of tabs) {
    if (tab.id) {
      chrome.tabs.sendMessage(tab.id, { type: "USER_CHANGED", userId }).catch(() => {
      });
    }
  }
}
async function getStats(userId) {
  try {
    const data = await apiRequest(`/api/sessions?user_id=${userId}`);
    return (data.sessions || []).map((s) => ({
      timestamp: new Date(s.started_at).getTime(),
      startTime: new Date(s.started_at).getTime(),
      endTime: s.ended_at ? new Date(s.ended_at).getTime() : new Date(s.started_at).getTime(),
      wpm: s.wpm,
      accuracy: s.accuracy,
      chars: s.total_characters,
      errors: s.errors,
      corrected: s.corrected,
      maxStreak: s.max_streak,
      videoId: s.video_id,
      videoTitle: s.video_title
    }));
  } catch (e) {
    console.log("[TypeTube] API unavailable for stats, using local storage");
    const statsKey = `typingStats_${userId}`;
    const result = await chrome.storage.local.get([statsKey]);
    return result[statsKey] || [];
  }
}
async function getPendingSessions() {
  const result = await chrome.storage.local.get("pendingSessions");
  return result.pendingSessions || [];
}
async function savePendingSessions(sessions) {
  await chrome.storage.local.set({ pendingSessions: sessions });
}
async function queueSession(session) {
  const pending = await getPendingSessions();
  const newSession = {
    ...session,
    id: `${session.user_id}_${session.started_at}_${Date.now()}`,
    queuedAt: Date.now()
  };
  pending.push(newSession);
  await savePendingSessions(pending);
  console.log("[TypeTube] Session queued for sync:", newSession.id);
}
async function syncPendingSessions() {
  const pending = await getPendingSessions();
  if (pending.length === 0) {
    return { synced: 0, failed: 0, remaining: 0 };
  }
  console.log("[TypeTube] Syncing", pending.length, "pending sessions...");
  const synced = [];
  let failed = 0;
  for (const session of pending) {
    try {
      await apiRequest("/api/sessions", {
        method: "POST",
        body: JSON.stringify({
          user_id: session.user_id,
          video_id: session.video_id,
          video_title: session.video_title,
          started_at: session.started_at,
          ended_at: session.ended_at,
          total_characters: session.total_characters,
          correct_characters: session.correct_characters,
          errors: session.errors,
          corrected: session.corrected,
          max_streak: session.max_streak,
          wpm: session.wpm,
          accuracy: session.accuracy,
          brainrot_factor: session.brainrot_factor
        })
      });
      synced.push(session.id);
      console.log("[TypeTube] Synced session:", session.id);
    } catch (e) {
      console.log("[TypeTube] Failed to sync session:", session.id, e);
      failed++;
      if (failed >= 3) {
        console.log("[TypeTube] Multiple failures, stopping sync attempt");
        break;
      }
    }
  }
  const remaining = pending.filter((s) => !synced.includes(s.id));
  await savePendingSessions(remaining);
  return { synced: synced.length, failed, remaining: remaining.length };
}
async function isApiAvailable() {
  try {
    activeApiBaseUrl = null;
    const url = await findWorkingApiServer();
    return url !== null;
  } catch {
    return false;
  }
}
async function getSyncStatus() {
  const pending = await getPendingSessions();
  const result = await chrome.storage.local.get("lastSyncAttempt");
  return {
    pendingCount: pending.length,
    lastSyncAttempt: result.lastSyncAttempt || null
  };
}
async function isMigrationComplete(userId) {
  const result = await chrome.storage.local.get(`migration_complete_${userId}`);
  return result[`migration_complete_${userId}`] === true;
}
async function markMigrationComplete(userId) {
  await chrome.storage.local.set({ [`migration_complete_${userId}`]: true });
}
async function getUserIdsWithLocalStats() {
  const allStorage = await chrome.storage.local.get(null);
  const userIds = [];
  for (const key of Object.keys(allStorage)) {
    if (key.startsWith("typingStats_")) {
      const userId = key.replace("typingStats_", "");
      if (userId && Array.isArray(allStorage[key]) && allStorage[key].length > 0) {
        userIds.push(userId);
      }
    }
  }
  return userIds;
}
async function migrateUserStats(userId) {
  const statsKey = `typingStats_${userId}`;
  const result = await chrome.storage.local.get([statsKey]);
  const stats = result[statsKey] || [];
  if (stats.length === 0) {
    return { migrated: 0, skipped: 0, failed: 0 };
  }
  console.log("[TypeTube] Migrating", stats.length, "sessions for user:", userId);
  let migrated = 0;
  let skipped = 0;
  let failed = 0;
  for (const stat of stats) {
    try {
      const sessionData = {
        user_id: userId,
        video_id: stat.videoId || null,
        video_title: stat.videoTitle || null,
        started_at: stat.startTime || stat.timestamp,
        ended_at: stat.endTime || stat.timestamp,
        total_characters: stat.chars || 0,
        correct_characters: (stat.chars || 0) - (stat.errors || 0),
        errors: stat.errors || 0,
        corrected: stat.corrected || 0,
        max_streak: stat.maxStreak || 0,
        wpm: stat.wpm || 0,
        accuracy: stat.accuracy || 0
      };
      await apiRequest("/api/sessions", {
        method: "POST",
        body: JSON.stringify(sessionData)
      });
      migrated++;
    } catch (e) {
      console.error("[TypeTube] Failed to migrate session:", e);
      failed++;
    }
  }
  return { migrated, skipped, failed };
}
async function migrateAllStats() {
  if (!await isApiAvailable()) {
    throw new Error("API server not available");
  }
  const userIds = await getUserIdsWithLocalStats();
  console.log("[TypeTube] Found users with local stats:", userIds);
  const userResults = {};
  let totalMigrated = 0;
  let totalFailed = 0;
  for (const userId of userIds) {
    if (await isMigrationComplete(userId)) {
      console.log("[TypeTube] Migration already complete for user:", userId);
      userResults[userId] = { migrated: 0, skipped: 0, failed: 0 };
      continue;
    }
    const result = await migrateUserStats(userId);
    userResults[userId] = result;
    totalMigrated += result.migrated;
    totalFailed += result.failed;
    if (result.failed === 0) {
      await markMigrationComplete(userId);
    }
  }
  return {
    success: totalFailed === 0,
    totalMigrated,
    totalFailed,
    userResults
  };
}
var BRAINROT_CACHE_KEY = "typetube_brainrot_cache";
var BRAINROT_CACHE_VERSION = 1;
async function loadBrainrotCache() {
  const result = await chrome.storage.local.get([BRAINROT_CACHE_KEY]);
  const cache = result[BRAINROT_CACHE_KEY];
  if (cache && cache.version === BRAINROT_CACHE_VERSION) {
    return cache;
  }
  return { version: BRAINROT_CACHE_VERSION, entries: {} };
}
async function saveBrainrotCache(cache) {
  await chrome.storage.local.set({ [BRAINROT_CACHE_KEY]: cache });
}
async function getCachedBrainrotFactor(videoId) {
  const cache = await loadBrainrotCache();
  const entry = cache.entries[videoId];
  if (entry) {
    const thirtyDaysMs = 30 * 24 * 60 * 60 * 1e3;
    if (Date.now() - entry.timestamp < thirtyDaysMs) {
      return entry.factor;
    }
  }
  return null;
}
async function cacheBrainrotFactor(videoId, factor) {
  const cache = await loadBrainrotCache();
  cache.entries[videoId] = { factor, timestamp: Date.now() };
  await saveBrainrotCache(cache);
}
async function getAnthropicApiKey() {
  try {
    const serverUrl = activeApiBaseUrl || await findWorkingApiServer();
    if (serverUrl) {
      const response = await fetch(`${serverUrl}/api/config`, {
        method: "GET",
        headers: { "Accept": "application/json" }
      });
      if (response.ok) {
        const data = await response.json();
        if (data.anthropicApiKey) {
          console.log("[TypeTube] Got API key from server");
          return data.anthropicApiKey;
        }
      }
    }
  } catch (e) {
    console.log("[TypeTube] Server unavailable for API key:", e);
  }
  const keyResult = await chrome.storage.local.get(["anthropicApiKey"]);
  if (keyResult.anthropicApiKey) {
    console.log("[TypeTube] Got API key from local storage");
    return keyResult.anthropicApiKey;
  }
  console.log("[TypeTube] No API key found anywhere");
  return null;
}
async function classifyVideosForBrainrot(videos) {
  const apiKey = await getAnthropicApiKey();
  if (!apiKey) {
    console.warn("[TypeTube] No Anthropic API key found - check server /api/config or extension settings");
    return videos.map((v) => ({ videoId: v.videoId, factor: 1, cached: false }));
  }
  console.log("[TypeTube] API key found, classifying", videos.length, "video(s)");
  const results = [];
  const videosToClassify = [];
  for (const video of videos) {
    const cached = await getCachedBrainrotFactor(video.videoId);
    if (cached !== null) {
      console.log(`[TypeTube] Using cached BF for "${video.title}": ${cached}x`);
      results.push({ videoId: video.videoId, factor: cached, cached: true });
    } else {
      videosToClassify.push(video);
    }
  }
  if (videosToClassify.length === 0) {
    return results;
  }
  const BATCH_SIZE = 10;
  const batches = [];
  for (let i = 0; i < videosToClassify.length; i += BATCH_SIZE) {
    batches.push(videosToClassify.slice(i, i + BATCH_SIZE));
  }
  console.log(`[TypeTube] Processing ${videosToClassify.length} videos in ${batches.length} batch(es)`);
  const batchPromises = batches.map((batch) => classifyVideoBatch(batch, apiKey));
  const batchResults = await Promise.all(batchPromises);
  for (const batchResult of batchResults) {
    for (const result of batchResult) {
      await cacheBrainrotFactor(result.videoId, result.factor);
      results.push(result);
    }
  }
  return results;
}
async function classifyVideoBatch(videos, apiKey) {
  let userMessage = `Classify these ${videos.length} YouTube videos. Return a JSON array with one object per video.

`;
  videos.forEach((video, index) => {
    userMessage += `Video ${index + 1} (ID: ${video.videoId}):
`;
    userMessage += `  Title: ${video.title}
`;
    userMessage += `  Channel: ${video.channelName}
`;
    if (video.duration) userMessage += `  Duration: ${video.duration}
`;
    userMessage += "\n";
  });
  userMessage += `
Return a JSON array like: [{"videoId": "...", "factor": 1.5}, ...]`;
  try {
    const response = await fetch("https://api.anthropic.com/v1/messages", {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        "x-api-key": apiKey,
        "anthropic-version": "2023-06-01",
        "anthropic-dangerous-direct-browser-access": "true"
      },
      body: JSON.stringify({
        model: "claude-haiku-4-5",
        max_tokens: 1e3,
        system: BRAINROT_SYSTEM_PROMPT + '\n\nWhen given multiple videos, return a JSON array with one object per video: [{"videoId": "...", "factor": X}, ...]',
        messages: [{ role: "user", content: userMessage }]
      })
    });
    if (!response.ok) {
      const error = await response.text();
      console.error("[TypeTube] Batch API error:", response.status, error);
      return videos.map((v) => ({ videoId: v.videoId, factor: 1, cached: false }));
    }
    const data = await response.json();
    const content = data.content?.[0]?.text || "";
    const jsonMatch = content.match(/\[[\s\S]*\]/);
    if (jsonMatch) {
      const parsed = JSON.parse(jsonMatch[0]);
      const validFactors = [0.25, 0.5, 0.75, 1, 1.25, 1.5, 2, 2.5, 4, 6, 10];
      const results = [];
      for (const item of parsed) {
        const video = videos.find((v) => v.videoId === item.videoId);
        if (!video) continue;
        let factor = parseFloat(item.factor) || 1;
        if (!validFactors.includes(factor)) {
          factor = validFactors.reduce(
            (prev, curr) => Math.abs(curr - factor) < Math.abs(prev - factor) ? curr : prev
          );
        }
        console.log(`[TypeTube] Batch classified "${video.title}": ${factor}x`);
        results.push({ videoId: item.videoId, factor, cached: false });
      }
      for (const video of videos) {
        if (!results.find((r) => r.videoId === video.videoId)) {
          console.log(`[TypeTube] Missing result for "${video.title}", defaulting to 1x`);
          results.push({ videoId: video.videoId, factor: 1, cached: false });
        }
      }
      return results;
    }
    console.error("[TypeTube] Could not parse batch response:", content);
    return videos.map((v) => ({ videoId: v.videoId, factor: 1, cached: false }));
  } catch (e) {
    console.error("[TypeTube] Batch classification error:", e);
    return videos.map((v) => ({ videoId: v.videoId, factor: 1, cached: false }));
  }
}
var SYNC_ALARM_NAME = "typetube-sync";
chrome.alarms.create(SYNC_ALARM_NAME, {
  periodInMinutes: 5
});
chrome.alarms.onAlarm.addListener(async (alarm) => {
  if (alarm.name === SYNC_ALARM_NAME) {
    const status = await getSyncStatus();
    if (status.pendingCount > 0) {
      await syncPendingSessions();
      await chrome.storage.local.set({ lastSyncAttempt: Date.now() });
    }
  }
});
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  switch (message.type) {
    case "GET_SETTINGS":
      chrome.storage.sync.get(Object.keys(DEFAULT_SETTINGS)).then(sendResponse);
      return true;
    case "SAVE_SETTINGS":
      chrome.storage.sync.set(message.settings).then(() => sendResponse({ success: true }));
      return true;
    case "GET_STATS":
      if (message.userId) {
        getStats(message.userId).then((stats) => {
          sendResponse({ typingStats: stats });
        });
      } else {
        sendResponse({ typingStats: [] });
      }
      return true;
    case "CLEAR_STATS":
      if (message.userId) {
        const statsKey = `typingStats_${message.userId}`;
        chrome.storage.local.set({ [statsKey]: [] }).then(() => sendResponse({ success: true }));
      } else {
        sendResponse({ error: "userId required" });
      }
      return true;
    case "GET_USER_LIST":
      getUsers().then((users) => {
        sendResponse({ users });
      });
      return true;
    case "GET_CURRENT_USER":
      getCurrentUser().then((user) => {
        sendResponse({ user });
      });
      return true;
    case "CREATE_USER":
      createUser(message.name, message.avatar).then((user) => {
        if (user) {
          sendResponse({ success: true, user });
        } else {
          sendResponse({ success: false, error: "Failed to create user" });
        }
      });
      return true;
    case "SWITCH_USER":
      switchUser(message.userId).then(() => {
        sendResponse({ success: true });
      });
      return true;
    // ============================================================================
    // Sync and Migration Message Handlers
    // ============================================================================
    case "QUEUE_SESSION":
      queueSession(message.session).then(() => {
        sendResponse({ success: true });
      }).catch((e) => {
        sendResponse({ success: false, error: e.message });
      });
      return true;
    case "SYNC_PENDING":
      syncPendingSessions().then((result) => {
        sendResponse({ success: true, ...result });
      }).catch((e) => {
        sendResponse({ success: false, error: e.message });
      });
      return true;
    case "GET_SYNC_STATUS":
      getSyncStatus().then((status) => {
        sendResponse(status);
      });
      return true;
    case "IS_API_AVAILABLE":
      isApiAvailable().then((available) => {
        sendResponse({ available });
      });
      return true;
    case "MIGRATE_DATA":
      migrateAllStats().then((result) => {
        sendResponse({ success: true, ...result });
      }).catch((e) => {
        sendResponse({ success: false, error: e.message });
      });
      return true;
    case "GET_MIGRATION_STATUS":
      (async () => {
        const userIds = await getUserIdsWithLocalStats();
        const migrationStatus = {};
        for (const userId of userIds) {
          migrationStatus[userId] = await isMigrationComplete(userId);
        }
        sendResponse({
          usersWithLocalData: userIds,
          migrationStatus,
          allMigrated: Object.values(migrationStatus).every((v) => v)
        });
      })();
      return true;
    // ============================================================================
    // Brainrot Factor Classification
    // ============================================================================
    case "GET_ANTHROPIC_API_KEY":
      (async () => {
        try {
          const serverUrl = activeApiBaseUrl || await findWorkingApiServer();
          if (serverUrl) {
            const response = await fetch(`${serverUrl}/api/config`, {
              method: "GET",
              headers: { "Accept": "application/json" }
            });
            if (response.ok) {
              const data = await response.json();
              if (data.anthropicApiKey) {
                sendResponse({ apiKey: data.anthropicApiKey, source: "server" });
                return;
              }
            }
          }
        } catch (e) {
        }
        const result = await chrome.storage.local.get(["anthropicApiKey"]);
        sendResponse({ apiKey: result.anthropicApiKey || null, source: result.anthropicApiKey ? "local" : null });
      })();
      return true;
    case "SET_ANTHROPIC_API_KEY":
      chrome.storage.local.set({ anthropicApiKey: message.apiKey }).then(() => {
        sendResponse({ success: true });
      });
      return true;
    case "CLASSIFY_VIDEOS":
      classifyVideosForBrainrot(message.videos).then((results) => {
        sendResponse({ success: true, results });
      }).catch((e) => {
        console.error("[TypeTube] Classification error:", e);
        sendResponse({ success: false, error: e.message });
      });
      return true;
    case "GET_CACHED_BRAINROT":
      getCachedBrainrotFactor(message.videoId).then((factor) => {
        sendResponse({ factor });
      });
      return true;
    default:
      sendResponse({ error: "Unknown message type" });
  }
});
